<?php
/**
 * Gelişmiş Dinamik Randevu Algoritması
 * Son güncelleme: 27.03.2025
 * 
 * Bu dosya, dinamik olarak uygun randevu saatlerini hesaplar.
 * Temel özellikleri:
 * - Temel zaman birimi 15 dakika olarak ayarlanmıştır
 * - Tüm hizmetler 15dk'nın katları olarak hesaplanır
 * - Periyot ayarı entegrasyonu ile esneklik sağlar
 * - Çalışma saatleri dikkate alınır
 * - Hizmet süreleri hesaba katılır
 * - Özel günler (tatiller, özel çalışma günleri) otomatik işlenir
 * - Gece mesaisi (gece yarısını aşan çalışma saatleri) düzgün hesaplanır
 * - Randevu çakışmaları engellenir
 * - Alternatif tarih önerileri sunar
 * - Önbellek mekanizması ile performansı artırır
 */

// Gerekli dosyaları dahil et
require_once $_SERVER['DOCUMENT_ROOT'] . '/admin/system/settings/db.php';
require_once $_SERVER['DOCUMENT_ROOT'] . '/admin/system/settings/eral.php';

/**
 * Loglama fonksiyonu (debug için)
 * @param string $message Kayıt edilecek mesaj
 * @param string $logType Log dosyası tipi
 */
function logRandevu($message, $logType = 'randevu_alg') {
    $logFile = $_SERVER['DOCUMENT_ROOT'] . '/admin/system/logs/' . $logType . '_' . date('Y-m-d') . '.log';
    $timestamp = date('Y-m-d H:i:s');
    file_put_contents($logFile, "[$timestamp] $message" . PHP_EOL, FILE_APPEND);
}

/**
 * Uygun Randevu Saatlerini Hesaplayan Sınıf
 * 
 * Bu sınıf, 15 dakikalık temel zaman birimi üzerinden çalışır.
 * Tüm randevu süreleri 15 dakikanın katları olarak hesaplanır.
 */
class RandevuAlgoritma {
    private $DB;
    private $settings;
    private $translations;
    private $useCache;
    private $usePeriotSetting;
    
    // Sabit değişkenler
    private const TEMEL_ZAMAN_BIRIMI_DK = 15; // Temel zaman birimi (dakika)
    private const CACHE_DIR = '/admin/system/cache/';
    private const CACHE_DURATION = 300; // Önbellek süresi (saniye) - 5 dakika
    
    /**
     * Kurucu method
     * @param PDO $DB Veritabanı bağlantısı
     * @param array $settings Sistem ayarları
     * @param array $translations Çeviriler
     */
    public function __construct($DB, $settings, $translations) {
        $this->DB = $DB;
        $this->settings = $settings;
        $this->translations = $translations;
        
        // Önbellek kullanımı aktif mi
        $this->useCache = isset($settings['use_timeslot_cache']) ? 
                           (bool)$settings['use_timeslot_cache'] : true;
        
        // Periyot ayarı aktif mi
        $this->usePeriotSetting = isset($settings['use_periot_setting']) ? 
                                  (bool)$settings['use_periot_setting'] : false;
        
        // Zaman dilimi ayarı
        date_default_timezone_set('Europe/Istanbul');
        
        // Log başlangıcı
        logRandevu("===== Randevu Algoritması Başlatıldı =====");
        logRandevu("Önbellek Kullanımı: " . ($this->useCache ? "Aktif" : "Pasif"));
        logRandevu("Periyot Ayarı: " . ($this->usePeriotSetting ? "Aktif" : "Pasif"));
    }
    
    /**
     * Uygun randevu saatlerini hesaplar ve döndürür
     * 
     * @param string $date Seçilen tarih (YYYY-MM-DD formatında)
     * @param int $providerId Hizmet sağlayıcı ID
     * @param int $serviceId Hizmet ID
     * @param array $unavailableHours Kullanılamayan saatler (varsa)
     * @param int|null $editId Düzenlenen randevu ID (varsa)
     * 
     * @return array Hesaplanan randevu saatleri
     */
    public function getAvailableTimes($date, $providerId, $serviceId, $unavailableHours = [], $editId = null) {
        // Temel değişkenler
        $suankiSaat = date("H:i");
        $bugunTarih = date('Y-m-d');
        
        // Başlangıç log kaydı
        logRandevu("İstek Parametreleri: Tarih: $date, ProviderID: $providerId, HizmetID: $serviceId");
        
        // Önbellekten veri al (eğer aktifse)
        if ($this->useCache && empty($editId)) {
            $cachedData = $this->getCachedTimeSlots($date, $providerId, $serviceId);
            if ($cachedData !== false) {
                logRandevu("Önbellekten veriler alındı");
                return $cachedData;
            }
        }
        
        // Adım 1: Dolu zaman dilimlerini al (15dk'lık birimlere dönüştürülmüş)
        $busyTimeSlots = $this->getBusyTimeSlots($providerId, $date, $editId);
        logRandevu("Dolu zaman dilimleri: " . count($busyTimeSlots) . " adet 15dk'lık birim");
        
        // Adım 2: Personel bilgilerini al
        $provider = $this->getProviderInfo($providerId);
        if (!$provider) {
            return ['error' => 'Personel bilgisi bulunamadı'];
        }
        
        // Adım 3: Hizmet bilgilerini al
        $service = $this->getServiceInfo($serviceId);
        if (!$service) {
            return ['error' => 'Hizmet bilgisi bulunamadı'];
        }
        $serviceDuration = (int)$service['hour']; // Dakika cinsinden hizmet süresi
        
        // Adım 4: Haftanın günü
        $weekDay = date('N', strtotime($date));
        
        // Adım 5: Özel gün kontrolü
        $specialDayInfo = $this->checkSpecialDay($date);
        $isSpecialDay = $specialDayInfo['isSpecialDay'] ?? false;
        $specialDay = $specialDayInfo['specialDay'] ?? null;
        
        // Adım 6: Çalışma saatlerini belirle (özel gün veya normal)
        if ($isSpecialDay) {
            $startTime = $specialDay['start_time'];
            $endTime = $specialDay['end_time'];
            
            // Özel gün bilgisi HTML'i
            $specialDayHtml = $this->generateSpecialDayHtml($specialDay, $date);
            logRandevu("Özel gün tespit edildi: {$specialDay['title']}, Saatler: $startTime - $endTime");
        } else {
            // Normal çalışma saatleri
            $workingTime = explode("-", $provider['workingTime']);
            $startTime = $workingTime[0];
            $endTime = $workingTime[1];
            $specialDayHtml = '';
            logRandevu("Normal çalışma saatleri: $startTime - $endTime");
        }
        
        // Adım 7: İzin günü kontrolü
        $breakDays = explode(",", $provider['breakTime']);
        if (in_array($weekDay, $breakDays)) {
            // İzin günü bildirimi
            return [
                'error' => 'dayoff',
                'html' => $this->generateDayoffHtml()
            ];
        }
        
        // Adım 8: Zaman dilimlerini hesapla (15dk'lık temel birimler)
        $availableSlots = $this->calculateAvailableTimeSlots(
            $date, 
            $startTime, 
            $endTime, 
            $serviceDuration, 
            $busyTimeSlots, 
            $bugunTarih, 
            $suankiSaat
        );
        
        // Adım 9: Eğer uygun zaman dilimi yoksa alternatif günler öner
        $alternativeSuggestions = [];
        $alternativeSuggestionsHtml = '';
        
        if (empty($availableSlots)) {
            $alternativeSuggestions = $this->suggestAlternativeDates($date, $providerId, $serviceId);
            $alternativeSuggestionsHtml = $this->formatAlternativeSuggestionsHtml($alternativeSuggestions);
            logRandevu("Alternatif tarih önerileri: " . count($alternativeSuggestions) . " adet");
        }
        
        // Adım 10: Sonuçları formatla ve döndür
        $result = [
            'slots' => $availableSlots,
            'specialDayHtml' => $specialDayHtml,
            'serviceDuration' => $serviceDuration,
            'alternativeSuggestionsHtml' => $alternativeSuggestionsHtml,
            'alternativeSuggestions' => $alternativeSuggestions
        ];
        
        // Sonuçları önbelleğe kaydet (eğer aktifse ve düzenleme yoksa)
        if ($this->useCache && empty($editId)) {
            $this->cacheTimeSlots($date, $providerId, $serviceId, $result);
        }
        
        return $result;
    }
    
    /**
     * Dolu zaman dilimlerini 15 dakikalık temel birimlere bölerek getirir ve 
     * ayrıca tüm randevuların başlangıç-bitiş saatlerini döndürür
     * 
     * @param int $providerId Hizmet sağlayıcı ID
     * @param string $date Tarih
     * @param int|null $editId Düzenlenen randevu ID (varsa)
     * 
     * @return array [busyTimeSlots, appointments] Dolu 15 dakikalık zaman dilimleri ve tüm randevular
     */
    private function getBusyTimeSlots($providerId, $date, $editId = null) {
        $busyTimeSlots = [];
        $appointments = []; // Tüm randevuların başlangıç-bitiş bilgilerini tutacak
        
        // Bugünün randevularını al
        $query = $this->DB->prepare("
            SELECT 
                id,
                appDate,
                startTime,
                endTime,
                service,
                coming as status
            FROM appointments 
            WHERE providerID = ? 
            AND DATE(appDate) = ? 
            AND coming != 4
        ");
        $query->execute([$providerId, $date]);
        
        while ($appointment = $query->fetch(PDO::FETCH_ASSOC)) {
            // Düzenlenen randevu ise atla
            if ($editId && $appointment['id'] == $editId) {
                continue;
            }
            
            // Hizmet süresini al
            $serviceQuery = $this->DB->prepare("SELECT hour FROM services WHERE id = ?");
            $serviceQuery->execute([$appointment['service']]);
            $serviceInfo = $serviceQuery->fetch(PDO::FETCH_ASSOC);
            $serviceDuration = (int)($serviceInfo['hour'] ?? 60); // Varsayılan 60 dakika
            
            // Log ekle
            logRandevu("Randevu ID: {$appointment['id']}, Hizmet ID: {$appointment['service']}, Süre: $serviceDuration dk");
            
            // Randevunun başlangıç saatini al
            $startTime = new DateTime($appointment['startTime']);
            $startHour = (int)$startTime->format('H');
            $startMinute = (int)$startTime->format('i');
            $startTimeInMinutes = $startHour * 60 + $startMinute;
            
            // Randevunun bitiş saatini al veya hesapla
            if (!empty($appointment['endTime'])) {
                $endTime = new DateTime($appointment['endTime']);
                $endHour = (int)$endTime->format('H');
                $endMinute = (int)$endTime->format('i');
                $endTimeInMinutes = $endHour * 60 + $endMinute;
            } else {
                // Bitiş saati yoksa, başlangıç + hizmet süresi
                $endTimeInMinutes = $startTimeInMinutes + $serviceDuration;
            }
            
            // Gece yarısını geçen randevular için düzeltme
            if ($endTimeInMinutes < $startTimeInMinutes) {
                $endTimeInMinutes += 24 * 60;
            }
            
            // Randevu bilgilerini ekle
            $appointments[] = [
                'id' => $appointment['id'],
                'start' => $startTimeInMinutes,
                'end' => $endTimeInMinutes,
                'duration' => $serviceDuration
            ];
            
            // Hizmet süresince her 15 dakikalık dilimi dolu olarak işaretle
            $slotCount = ceil($serviceDuration / self::TEMEL_ZAMAN_BIRIMI_DK);
            for ($i = 0; $i < $slotCount; $i++) {
                $slotStartMinute = $startTimeInMinutes + ($i * self::TEMEL_ZAMAN_BIRIMI_DK);
                
                // 24 saat formatına göre modülo hesapla
                $slotRealMinute = $slotStartMinute % (24 * 60);
                
                // Bu zaman dilimini dolu olarak işaretle
                $slotHour = floor($slotRealMinute / 60);
                $slotMinute = $slotRealMinute % 60;
                
                // Formatlı saat oluştur (HH:MM)
                $slotTime = sprintf("%02d:%02d", $slotHour, $slotMinute);
                $busyTimeSlots[] = $slotTime;
            }
        }
        
        // Çalışılmayacak saatleri "saatler" tablosundan al
        $unavailableHoursQuery = $this->DB->prepare("
            SELECT 
                saat, 
                tarih
            FROM saatler 
            WHERE provid = ? 
            AND tarih = ?
        ");
        $unavailableHoursQuery->execute([$providerId, $date]);
        
        logRandevu("Çalışılmayacak saatlerin kontrolü başlıyor - Tarih: $date, ProviderID: $providerId");
        
        // Tüm çalışılmayacak saatleri al ve işle
        $unavailableHours = [];
        while ($hour = $unavailableHoursQuery->fetch(PDO::FETCH_ASSOC)) {
            // Saat formatını standardize et (HH:MM)
            $timeStr = $hour['saat'];
            if (strlen($timeStr) > 5) {  // HH:MM:SS formatındaysa
                $timeStr = substr($timeStr, 0, 5);  // İlk 5 karakteri al (HH:MM)
            }
            
            // Saat ve dakikayı ayrıştır
            $timeParts = explode(':', $timeStr);
            $hourValue = (int)$timeParts[0];
            $minuteValue = isset($timeParts[1]) ? (int)$timeParts[1] : 0;
            
            // Dakika değerine çevir
            $timeInMinutes = $hourValue * 60 + $minuteValue;
            
            // Bu saati tüm 15 dakikalık dilimleri kapsayacak şekilde işaretle
            // Periyot aralığına bağlı olarak bloklar halinde işaretle
            $periyot = intval($this->settings['periot'] ?? 45);
            $blockSize = $periyot > 0 ? $periyot : 45; // Dakika cinsinden çalışılmayacak blok boyutu
            
            // Bloğu kapsayan tüm 15 dakikalık dilimleri işaretle
            for ($minute = 0; $minute < $blockSize; $minute += self::TEMEL_ZAMAN_BIRIMI_DK) {
                $currentMinute = $timeInMinutes + $minute;
                $currentHour = floor($currentMinute / 60);
                $currentMinuteValue = $currentMinute % 60;
                
                // 24 saat formatında modülo
                if ($currentHour >= 24) {
                    $currentHour %= 24;
                }
                
                // Formatlı saat:dakika
                $formattedTime = sprintf("%02d:%02d", $currentHour, $currentMinuteValue);
                $busyTimeSlots[] = $formattedTime;
                
                logRandevu("Çalışılmayacak saat bloğu eklendi: $timeStr -> $formattedTime");
            }
            
            // Ayrıca, bu çalışılmayacak saati takip eden dakikalar boyunca 
            // hizmet süresi kadar olan başlangıç saatlerini de bloklayalım
            // Böylece bir sonraki periyoda geçene kadar olan tüm başlangıç saatleri de engellenmiş olur
            
            // Hizmet süresini al (herhangi bir değer alıyoruz, sonra hesaplama yaparken kullanacağız)
            $serviceQuery = $this->DB->prepare("SELECT AVG(hour) as avg_duration FROM services WHERE id > 0");
            $serviceQuery->execute();
            $serviceInfo = $serviceQuery->fetch(PDO::FETCH_ASSOC);
            $avgServiceDuration = (int)($serviceInfo['avg_duration'] ?? 30); // Ortalama hizmet süresi
            
            // İlave blok: Çalışılmayacak saat + periyot kadar süre boyunca
            // başlangıç saatlerini engelle (böylece kapatılan saatin etkisi tam olur)
            for ($minute = $blockSize; $minute < $blockSize + $avgServiceDuration; $minute += self::TEMEL_ZAMAN_BIRIMI_DK) {
                $currentMinute = $timeInMinutes + $minute - $avgServiceDuration; // Hizmetin başlangıç noktasını hesapla
                
                // Eğer başlangıç noktası hala bloğun içindeyse atla
                if ($minute - $avgServiceDuration < 0) {
                    continue;
                }
                
                $currentHour = floor($currentMinute / 60);
                $currentMinuteValue = $currentMinute % 60;
                
                // 24 saat formatında modülo
                if ($currentHour >= 24) {
                    $currentHour %= 24;
                }
                
                // Formatlı saat:dakika
                $formattedTime = sprintf("%02d:%02d", $currentHour, $currentMinuteValue);
                if (!in_array($formattedTime, $busyTimeSlots)) {
                    $busyTimeSlots[] = $formattedTime;
                    logRandevu("Ek blok - Çalışılmayacak saat sonrası: $timeStr -> $formattedTime");
                }
            }
        }
        
        // Hem dolu zaman dilimlerini hem de randevu bilgilerini döndür
        return [
            'busyTimeSlots' => $busyTimeSlots,
            'appointments' => $appointments
        ];
    }
    
    /**
     * Belirli bir tarih için hızlı uygunluk kontrolü yapar
     * (Alternatif tarih önerileri için kullanılır)
     * 
     * @param string $date Tarih
     * @param int $providerId Hizmet sağlayıcı ID
     * @param int $serviceId Hizmet ID
     * 
     * @return array Uygun zaman dilimlerinin sayısı
     */
    private function getQuickAvailabilityCheck($date, $providerId, $serviceId) {
        // Önbellekten veri al (eğer aktifse)
        if ($this->useCache) {
            $cachedData = $this->getCachedTimeSlots($date, $providerId, $serviceId);
            if ($cachedData !== false) {
                return $cachedData['slots'] ?? [];
            }
        }
        
        // Hızlı kontrol için temel hesaplamaları yap
        $busyTimeSlots = $this->getBusyTimeSlots($providerId, $date);
        $provider = $this->getProviderInfo($providerId);
        $service = $this->getServiceInfo($serviceId);
        
        if (!$provider || !$service) {
            return [];
        }
        
        $serviceDuration = (int)$service['hour'];
        
        // Özel gün kontrolü
        $specialDayInfo = $this->checkSpecialDay($date);
        $isSpecialDay = $specialDayInfo['isSpecialDay'] ?? false;
        $specialDay = $specialDayInfo['specialDay'] ?? null;
        
        // Çalışma saatlerini belirle
        if ($isSpecialDay) {
            $startTime = $specialDay['start_time'];
            $endTime = $specialDay['end_time'];
        } else {
            $workingTime = explode("-", $provider['workingTime']);
            $startTime = $workingTime[0];
            $endTime = $workingTime[1];
        }
        
        // Haftanın günü
        $weekDay = date('N', strtotime($date));
        $breakDays = explode(",", $provider['breakTime']);
        
        // İzin günü ise boş döndür
        if (in_array($weekDay, $breakDays)) {
            return [];
        }
        
        // Zaman dilimlerini hesapla
        return $this->calculateAvailableTimeSlots(
            $date, 
            $startTime, 
            $endTime, 
            $serviceDuration, 
            $busyTimeSlots, 
            date('Y-m-d'), // Bugünün tarihi
            date('H:i')    // Şu anki saat
        );
    }
    
    /**
     * Personel bilgilerini getirir
     * 
     * @param int $providerId Personel ID
     * 
     * @return array|false Personel bilgileri
     */
    private function getProviderInfo($providerId) {
        $query = $this->DB->prepare("SELECT * FROM provider WHERE id = ?");
        $query->execute([$providerId]);
        return $query->fetch(PDO::FETCH_ASSOC);
    }
    
    /**
     * Hizmet bilgilerini getirir
     * 
     * @param int $serviceId Hizmet ID
     * 
     * @return array|false Hizmet bilgileri
     */
    private function getServiceInfo($serviceId) {
        $query = $this->DB->prepare("SELECT * FROM services WHERE id = ?");
        $query->execute([$serviceId]);
        return $query->fetch(PDO::FETCH_ASSOC);
    }
    
    /**
     * Özel gün kontrolü yapar
     * 
     * @param string $date Kontrol edilecek tarih
     * 
     * @return array Özel gün bilgileri
     */
    private function checkSpecialDay($date) {
        $result = [
            'isSpecialDay' => false,
            'specialDay' => null
        ];
        
        try {
            logRandevu("Özel gün kontrolü yapılıyor: $date");
            
            // Tablo var mı?
            $checkTable = $this->DB->query("SHOW TABLES LIKE 'special_days'");
            if ($checkTable->rowCount() == 0) {
                logRandevu("Özel günler tablosu bulunamadı");
                return $result;
            }
            
            // Ayar aktif mi?
            $settingsCheck = $this->DB->prepare("SELECT specialDaysSystem FROM settings WHERE id = 1");
            $settingsCheck->execute();
            $settingsRow = $settingsCheck->fetch(PDO::FETCH_ASSOC);
            $specialDaysSystemSetting = isset($settingsRow['specialDaysSystem']) ? $settingsRow['specialDaysSystem'] : '0';
            
            if ($specialDaysSystemSetting != '1') {
                logRandevu("Özel günler sistemi kapalı");
                return $result;
            }
            
            // Özel gün kontrolü
            $specialDayCheck = $this->DB->prepare("
                SELECT * FROM special_days 
                WHERE ? BETWEEN start_date AND end_date 
                AND status = 1
            ");
            $specialDayCheck->execute([$date]);
            
            if ($specialDayCheck->rowCount() > 0) {
                $specialDay = $specialDayCheck->fetch(PDO::FETCH_ASSOC);
                logRandevu("Özel gün bulundu: " . $specialDay['title']);
                
                $result = [
                    'isSpecialDay' => true,
                    'specialDay' => $specialDay
                ];
            }
            
            return $result;
        } catch (Exception $e) {
            logRandevu("Özel gün kontrolünde hata: " . $e->getMessage());
            return $result;
        }
    }
    
    /**
     * Özel gün bilgilendirme HTML'i oluşturur
     * 
     * @param array $specialDay Özel gün bilgileri
     * @param string $date Tarih
     * 
     * @return string HTML çıktısı
     */
    private function generateSpecialDayHtml($specialDay, $date) {
        $html = '<div class="alert alert-info mb-2" style="font-size: 14px;">';
        $html .= '<strong>' . $specialDay['title'] . '</strong> nedeniyle bugün çalışma saatleri: <strong>' . 
                 $specialDay['start_time'] . ' - ' . $specialDay['end_time'] . 
                 '</strong> olarak uygulanmaktadır. <small>(Özel gün saatleri <i class="fas fa-star-of-life" style="font-size: 7px; color: #0077be;"></i> işareti ile belirtilmiştir)</small>';
        
        // Gece çalışması kontrolü
        $specialStartHour = intval(substr($specialDay['start_time'], 0, 2));
        $specialEndHour = intval(substr($specialDay['end_time'], 0, 2));
        $isOvernight = $specialEndHour < $specialStartHour;
        
        if ($isOvernight) {
            $html .= '<div class="mt-1"><small class="text-muted">Not: Gece yarısından sonraki saatler (00:00 sonrası) ertesi gün takviminde görüntülenecektir.</small></div>';
            
            // Ertesi gün tarihini göster
            $nextDayFormatted = date('d.m.Y', strtotime($date . ' +1 day'));
            $html .= '<div class="mt-1"><small class="text-info"><strong>Önemli:</strong> ' . 
                     $nextDayFormatted . ' tarihini seçerek saat 01:00-' . $specialDay['end_time'] . 
                     ' arası özel gün saatlerini görebilirsiniz.</small></div>';
        }
        
        $html .= '</div>';
        return $html;
    }
    
    /**
     * İzin günü uyarı HTML'i oluşturur
     * 
     * @return string HTML çıktısı
     */
    private function generateDayoffHtml() {
        $html = '<script>
            $(document).ready(function() {
                Swal.fire({
                    title: "' . $this->translations['error'] . '",
                    text: "' . $this->translations['dayoff'] . '",
                    icon: "warning",
                    confirmButtonColor: "#7367f0",
                    confirmButtonText: "' . $this->translations['okey'] . '"
                });
            });
        </script>';
        
        return $html;
    }
    
    /**
     * Kullanılabilir zaman dilimlerini hesaplar
     * 
     * @param string $date Tarih
     * @param string $startTime Başlangıç saati
     * @param string $endTime Bitiş saati
     * @param int $serviceDuration Hizmet süresi (dakika)
     * @param array $busyTimeSlots Dolu 15dk'lık zaman dilimleri ve mevcut randevular 
     * @param string $today Bugünün tarihi
     * @param string $currentTime Şu anki saat
     * 
     * @return array Kullanılabilir zaman dilimleri
     */
    private function calculateAvailableTimeSlots($date, $startTime, $endTime, $serviceDuration, $busyTimeData, $today, $currentTime) {
        // Dolu zaman dilimleri ve mevcut randevu bilgilerini ayır
        $busyTimeSlots = $busyTimeData['busyTimeSlots'] ?? $busyTimeData;
        $existingAppointments = $busyTimeData['appointments'] ?? [];
        
        // Mesai saatlerini dakikaya çevir
        $startHour = intval(substr($startTime, 0, 2));
        $startMinute = intval(substr($startTime, 3, 2));
        $startTimeInMinutes = $startHour * 60 + $startMinute;
        
        $endHour = intval(substr($endTime, 0, 2));
        $endMinute = intval(substr($endTime, 3, 2));
        $endTimeInMinutes = $endHour * 60 + $endMinute;
        
        // Gece vardiyası kontrolü (örneğin 22:00-06:00)
        $isOvernight = $endTimeInMinutes <= $startTimeInMinutes;
        if ($isOvernight) {
            logRandevu("Gece vardiyası tespit edildi: $startTime - $endTime");
            $endTimeInMinutes += 24 * 60; // 24 saat ekliyoruz
        }
        
        // Bugün için şimdiki saati dakikaya çevir
        $currentHourInMinutes = 0;
        if ($date == $today) {
            $currentHour = intval(substr($currentTime, 0, 2));
            $currentMinute = intval(substr($currentTime, 3, 2));
            $currentHourInMinutes = $currentHour * 60 + $currentMinute;
            logRandevu("Bugünün tarihi: Şimdiki saat dakika cinsinden: $currentHourInMinutes");
        }
        
        // Hizmet süresini 15dk'lık birim sayısına çevir
        $serviceSlotsNeeded = ceil($serviceDuration / self::TEMEL_ZAMAN_BIRIMI_DK);
        logRandevu("Hizmet süresi: $serviceDuration dk, Gereken 15dk'lık birim sayısı: $serviceSlotsNeeded");
        
        // Periyot ayarını al (dakika olarak)
        $periyot = intval($this->settings['periot'] ?? 15);
        if ($periyot <= 0) {
            $periyot = 15; // Varsayılan periyot 15 dakika
        }
        
        // Adım miktarını belirle - Hizmet süresine göre hesapla
        $stepSize = self::TEMEL_ZAMAN_BIRIMI_DK; // Her zaman 15 dakikalık adımlar kullan
        logRandevu("Adım miktarı: $stepSize dakika (Periyot ayarı: " . ($this->usePeriotSetting ? "Aktif" : "Pasif") . ")");
        
        // Uygun zaman dilimleri listesi
        $availableSlots = [];
        
        // Mesai başlangıcından bitimine kadar, 15 dakikalık adımlarla ilerle
        for ($totalMinutes = $startTimeInMinutes; $totalMinutes <= $endTimeInMinutes - $serviceDuration; $totalMinutes += $stepSize) {
            // Gerçek dakikayı modülo ile hesapla (24 saat formatı)
            $realMinutes = $totalMinutes;
            if ($realMinutes >= 24 * 60) {
                $realMinutes = $realMinutes % (24 * 60);
            }
            
            // Başlangıç saati ve dakikası
            $hour = floor($realMinutes / 60);
            $minute = $realMinutes % 60;
            
            // Formatlı saat:dakika
            $formattedStartTime = sprintf("%02d:%02d", $hour, $minute);
            
            // Gece saati mi (00:00-05:59 arası)?
            $isLateHour = ($hour >= 0 && $hour < 6);
            
            // Bugün için geçmiş saat mi?
            if ($date == $today && $realMinutes <= $currentHourInMinutes) {
                continue; // Geçmiş saati atla
            }
            
            // Hizmet bitiş zamanını hesapla
            $serviceEndMinutes = $totalMinutes + $serviceDuration;
            $serviceEndRealMinutes = $serviceEndMinutes;
            if ($serviceEndRealMinutes >= 24 * 60) {
                $serviceEndRealMinutes = $serviceEndRealMinutes % (24 * 60);
            }
            $serviceEndHour = floor($serviceEndRealMinutes / 60);
            $serviceEndMinute = $serviceEndRealMinutes % 60;
            $formattedEndTime = sprintf("%02d:%02d", $serviceEndHour, $serviceEndMinute);
            
            // Bu zaman dilimi için gereken tüm 15 dakikalık birimler uygun mu?
            $isAvailable = true;
            
            // BÖLÜM 1: Mevcut tüm randevularla çakışma kontrolü (daha kesin ve net)
            if (!empty($existingAppointments)) {
                foreach ($existingAppointments as $appt) {
                    // Çakışma durumları:
                    // 1. Yeni randevunun başlangıcı, mevcut randevunun içinde
                    // 2. Yeni randevunun bitişi, mevcut randevunun içinde
                    // 3. Yeni randevu, mevcut randevuyu tamamen kapsıyor
                    if (($totalMinutes >= $appt['start'] && $totalMinutes < $appt['end']) || 
                        ($serviceEndMinutes > $appt['start'] && $serviceEndMinutes <= $appt['end']) ||
                        ($totalMinutes <= $appt['start'] && $serviceEndMinutes >= $appt['end'])) {
                        $isAvailable = false;
                        logRandevu("Saat $formattedStartTime - $formattedEndTime, mevcut randevu ile çakışıyor (Randevu ID: {$appt['id']})");
                        break;
                    }
                }
                
                // Eğer çakışma varsa, sonraki saate geç
                if (!$isAvailable) {
                    continue;
                }
            }
            
            // BÖLÜM 2: Tekil zaman dilimlerinin doluluğunu kontrol et (eski yöntem)
            // Hizmet süresi boyunca tüm 15 dakikalık birimleri kontrol et
            for ($i = 0; $i < $serviceSlotsNeeded; $i++) {
                $slotMinutes = $totalMinutes + ($i * self::TEMEL_ZAMAN_BIRIMI_DK);
                $slotRealMinutes = $slotMinutes % (24 * 60);
                $slotHour = floor($slotRealMinutes / 60);
                $slotMinute = $slotRealMinutes % 60;
                $slotFormatted = sprintf("%02d:%02d", $slotHour, $slotMinute);
                
                // Bu 15 dakikalık dilim dolu mu?
                if (in_array($slotFormatted, $busyTimeSlots)) {
                    $isAvailable = false;
                    logRandevu("Saat $formattedStartTime - 15dk dilimlerinde dolu zaman dilimi var: $slotFormatted");
                    break;
                }
            }
            
            // Uygun ise listeye ekle
            if ($isAvailable) {
                $availableSlots[] = [
                    "startTime" => $formattedStartTime,
                    "endTime" => $formattedEndTime,
                    "isLateHour" => $isLateHour,
                    "duration" => $serviceDuration
                ];
            }
        }
        
        // Gece saatlerini başa alma
        if ($isOvernight) {
            usort($availableSlots, function($a, $b) {
                // Gece yarısından sonraki saatler (00:00-05:59) önce gelsin
                $isEarlyHourA = $a["isLateHour"];
                $isEarlyHourB = $b["isLateHour"];
                
                if ($isEarlyHourA && !$isEarlyHourB) {
                    return -1; // A önce gelsin
                } else if (!$isEarlyHourA && $isEarlyHourB) {
                    return 1; // B önce gelsin
                } else {
                    // İkisi de gece yarısından sonra ya da ikisi de gece yarısından önce ise normal sıralama
                    return strcmp($a["startTime"], $b["startTime"]);
                }
            });
        }
        
        return $availableSlots;
    }
    
    /**
     * Alternatif tarih önerileri oluşturur
     * 
     * @param string $date Seçilen tarih
     * @param int $providerId Personel ID
     * @param int $serviceId Hizmet ID
     * 
     * @return array Önerilen tarihler
     */
    private function suggestAlternativeDates($date, $providerId, $serviceId) {
        $suggestions = [];
        $currentDate = new DateTime($date);
        
        // İleri doğru 5 gün için kontrol et
        for ($i = 1; $i <= 5; $i++) {
            $checkDate = clone $currentDate;
            $checkDate->modify("+$i day");
            $formattedDate = $checkDate->format('Y-m-d');
            
            // Bu tarih için uygun saatleri hesapla
            $availableSlots = $this->getQuickAvailabilityCheck($formattedDate, $providerId, $serviceId);
            
            if (!empty($availableSlots)) {
                $suggestions[] = [
                    'date' => $formattedDate,
                    'formatted_date' => $checkDate->format('d.m.Y'),
                    'day_name' => $this->getDayName($checkDate->format('N')),
                    'available_count' => count($availableSlots)
                ];
                
                // En fazla 3 öneri yeterli
                if (count($suggestions) >= 3) {
                    break;
                }
            }
        }
        
        return $suggestions;
    }
    
    /**
     * Alternatif tarih önerilerini HTML olarak formatlar
     * 
     * @param array $suggestions Önerilen tarihler
     * 
     * @return string HTML çıktısı
     */
    private function formatAlternativeSuggestionsHtml($suggestions) {
        if (empty($suggestions)) {
            return '';
        }
        
        $html = '<div class="alert alert-info mt-3">';
        $html .= '<h5><i class="fas fa-calendar-alt me-2"></i>Alternatif Tarih Önerileri</h5>';
        $html .= '<div class="row mt-2">';
        
        foreach ($suggestions as $suggestion) {
            $html .= '<div class="col-md-4 mb-2">';
            $html .= '<a href="javascript:void(0)" class="alternate-date-suggestion" data-date="' . $suggestion['formatted_date'] . '">';
            $html .= '<div class="card border-primary">';
            $html .= '<div class="card-body p-2 text-center">';
            $html .= '<strong>' . $suggestion['formatted_date'] . '</strong> (' . $suggestion['day_name'] . ')<br>';
            $html .= '<small>' . $suggestion['available_count'] . ' uygun saat</small>';
            $html .= '</div></div></a></div>';
        }
        
        $html .= '</div></div>';
        
        // JavaScript ekle
        $html .= '<script>
            $(document).ready(function() {
                $(".alternate-date-suggestion").click(function() {
                    var date = $(this).data("date");
                    $("#start-date").val(date).trigger("change");
                });
            });
        </script>';
        
        return $html;
    }
    
    /**
     * Gün numarasına göre gün adını döndürür
     * 
     * @param int $dayNumber Gün numarası (1-7)
     * 
     * @return string Gün adı
     */
    private function getDayName($dayNumber) {
        $days = [
            1 => 'Pazartesi',
            2 => 'Salı',
            3 => 'Çarşamba',
            4 => 'Perşembe',
            5 => 'Cuma',
            6 => 'Cumartesi',
            7 => 'Pazar'
        ];
        
        return $days[$dayNumber] ?? '';
    }
    
    /**
     * Önbellekten veri getirir
     * 
     * @param string $date Tarih
     * @param int $providerId Personel ID
     * @param int $serviceId Hizmet ID
     * 
     * @return array|false Önbellekteki veri
     */
    private function getCachedTimeSlots($date, $providerId, $serviceId) {
        // Önbellek anahtarı oluştur
        $cacheKey = "timeslots_{$date}_{$providerId}_{$serviceId}";
        
        // Önbellek dosyası
        $cacheDir = $_SERVER['DOCUMENT_ROOT'] . self::CACHE_DIR;
        $cacheFile = $cacheDir . $cacheKey . '.json';
        
        // Önbellekte var mı kontrol et
        if (file_exists($cacheFile) && (time() - filemtime($cacheFile) < self::CACHE_DURATION)) {
            return json_decode(file_get_contents($cacheFile), true);
        }
        
        return false;
    }
    
    /**
     * Verileri önbelleğe kaydeder
     * 
     * @param string $date Tarih
     * @param int $providerId Personel ID
     * @param int $serviceId Hizmet ID
     * @param array $data Kaydedilecek veri
     */
    private function cacheTimeSlots($date, $providerId, $serviceId, $data) {
        // Önbellek anahtarı oluştur
        $cacheKey = "timeslots_{$date}_{$providerId}_{$serviceId}";
        
        // Önbellek dizini
        $cacheDir = $_SERVER['DOCUMENT_ROOT'] . self::CACHE_DIR;
        
        // Dizin yoksa oluştur
        if (!is_dir($cacheDir)) {
            if (!mkdir($cacheDir, 0755, true)) {
                logRandevu("HATA: Önbellek dizini oluşturulamadı: " . $cacheDir);
                return;
            }
        }
        
        // Veriyi kaydet
        $cacheFile = $cacheDir . $cacheKey . '.json';
        file_put_contents($cacheFile, json_encode($data));
    }
    
    /**
     * Önbelleği temizler
     * 
     * @param string|null $date Belirli bir tarih için önbelleği temizle (null ise tüm önbellek)
     */
    public function clearCache($date = null) {
        $cacheDir = $_SERVER['DOCUMENT_ROOT'] . self::CACHE_DIR;
        
        if (!is_dir($cacheDir)) {
            return;
        }
        
        $files = scandir($cacheDir);
        foreach ($files as $file) {
            if ($file === '.' || $file === '..') {
                continue;
            }
            
            $filePath = $cacheDir . $file;
            
            // Belirli bir tarih için mi temizlenecek?
            if ($date !== null) {
                if (strpos($file, "timeslots_{$date}_") === 0) {
                    unlink($filePath);
                }
            } else {
                // Tüm önbelleği temizle
                unlink($filePath);
            }
        }
    }
    
    /**
     * Zaman dilimlerini HTML olarak formatlar
     * 
     * @param array $slots Zaman dilimleri
     * @param int $serviceDuration Hizmet süresi
     * 
     * @return string HTML çıktısı
     */
    public function formatTimeSlotsHtml($slots, $serviceDuration) {
        $html = '<div class="row custom-options-checkable g-1" style="display: flex; flex-wrap: wrap;">';
        
        if (count($slots) == 0) {
            $html .= '<div class="col-12 alert alert-warning">Bu tarihte uygun randevu saati bulunmamaktadır.</div>';
            logRandevu("Uyarı: Hiç uygun randevu saati bulunamadı!");
        } else {
            $i = 0;
            foreach ($slots as $slot) {
                $startTime = $slot['startTime'];
                $endTime = $slot['endTime'];
                $isLateHour = $slot['isLateHour'];
                
                // CSS sınıfı oluştur
                $cssClass = $isLateHour ? ' late-hour' : '';
                
                $html .= '<div class="col-md-4 mb-2 p-1">
                    <input type="radio" class="custom-option-item-check' . $cssClass . '"
                           id="timeOption'.$i.'"
                           name="customOptionsCheckableRadiosWithIcon"
                           value="'.$startTime.':00">
                    <label class="custom-option-item text-center p-2" for="timeOption'.$i.'">
                        <strong style="font-size: 1.1rem">'.$startTime.' - '.$endTime.'</strong>
                        <small class="d-block mt-1" style="font-size: 0.85rem;">'.$serviceDuration.' dakika</small>
                    </label>
                </div>';
                
                $i++;
            }
            
            logRandevu("Toplam " . count($slots) . " adet uygun randevu saati bulundu.");
        }
        
        $html .= '</div>';
        
        // JavaScript eklentisi
        $html .= '<script>
        document.addEventListener("DOMContentLoaded", function() {
            // Radio buttonları seç
            var radios = document.querySelectorAll(".custom-option-item-check");
            
            // Her birine click event listener ekle
            radios.forEach(function(radio) {
                radio.addEventListener("change", function() {
                    // Önce tüm etiketleri pasif yap
                    document.querySelectorAll(".custom-option-item").forEach(function(label) {
                        label.classList.remove("active");
                    });
                    
                    // Seçilen etiketi aktif yap
                    this.nextElementSibling.classList.add("active");
                    
                    // Gizli input alanına değeri yaz
                    var selectedTimeInput = document.getElementById("selectedTimeInput");
                    if (selectedTimeInput) {
                        selectedTimeInput.value = this.value;
                        console.log("Saat seçildi: " + this.value);
                    }
                });
            });
        });
        </script>';
        
        return $html;
    }
}

// POST verilerini logla
logRandevu("POST verileri: " . print_r($_POST, true));

// API Çağrısı
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Değişkenleri al
    date_default_timezone_set('Europe/Istanbul');
    $suankiSaat = date("H:i");
    $bugunTarih = date('Y-m-d');
    
    if (empty($_POST['providerID']) || empty($_POST['date'])) {
        echo '<div class="alert alert-danger">Eksik parametreler: providerID ve date gerekli.</div>';
        exit;
    }
    
    $providerID = $_POST['providerID'];
    $dateString = $_POST['date'];
    $date = date("Y-m-d", strtotime($dateString));
    $hizmetID = isset($_POST['hizmet']) ? $_POST['hizmet'] : null;
    $editId = isset($_POST['edit']) ? $_POST['edit'] : null;
    
    // Bir yönlendirme yapalım
    logRandevu("RandevuAlgoritma'ya istek geldi, AvailableTimeAjax.php'ye yönlendiriliyor. ProviderID: $providerID, Tarih: $date, Hizmet: $hizmetID");
    
    // POST verilerini doğrudan diğer dosyaya aktaralım
    $_POST['redirect_from_algorithm'] = 'true';
    
    // AvailableTimeAjax.php dosyasını include edelim
    include_once __DIR__ . '/AvailableTimeAjax.php';
    exit;
    
    // Aşağıdaki kod artık çalışmayacak, yönlendirme yapıyoruz
    
    // Algoritma sınıfını başlat
    $randevuAlgoritma = new RandevuAlgoritma($DB, $settings, $translations);
    
    try {
        // Hizmet kontrolü
        if (!$hizmetID) {
            echo '<div class="alert alert-danger">Hizmet bilgisi bulunamadı. Lütfen sayfayı yenileyip tekrar deneyin.</div>';
            exit;
        }
        
        // Çalışılmayacak saatler (tarih ve saat birleşik format)
        $unavailableHours = [];
        if (!empty($_POST['unavailableHours'])) {
            $unavailableHours = json_decode($_POST['unavailableHours'], true);
        }
        
        // Uygun saatleri hesapla
        $result = $randevuAlgoritma->getAvailableTimes(
            $date, 
            $providerID, 
            $hizmetID, 
            $unavailableHours, 
            $editId
        );
        
        // Sonucu işle
        if (isset($result['error'])) {
            if ($result['error'] === 'dayoff' && isset($result['html'])) {
                // İzin günü bildirimi
                echo $result['html'];
            } else {
                // Genel hata
                echo '<div class="alert alert-danger">' . $result['error'] . '</div>';
            }
        } else {
            // Özel gün bilgilendirmesi (varsa)
            if (!empty($result['specialDayHtml'])) {
                echo $result['specialDayHtml'];
            }
            
            // Zaman dilimlerini göster
            echo $randevuAlgoritma->formatTimeSlotsHtml(
                $result['slots'],
                $result['serviceDuration']
            );
            
            // Alternatif tarih önerileri (eğer uygun saat yoksa)
            if (empty($result['slots']) && !empty($result['alternativeSuggestionsHtml'])) {
                echo $result['alternativeSuggestionsHtml'];
            }
        }
        
        // Takvim Break sorgulaması - eski sistemden kopyalandı
        if (!empty($_POST['providerID'])) {
            $breakCheck = $DB->prepare("SELECT * from provider where id=? and breakstatus=?");
            $breakCheck->execute(array($_POST['providerID'], '1'));
            
            if ($breakCheck->rowCount()) {
                $break = $breakCheck->fetch(PDO::FETCH_ASSOC);
                ?>
<script type="text/javascript">
$(document).ready(function() {
  var start_time = "<?=$break['break_start_time']?>";
  var end_time = "<?=$break['break_end_time']?>";
  startDate = $('#start-date');
  if (startDate.length) {
    var start = startDate.flatpickr({
      disable: [{
        from: start_time,
        to: end_time
      }],
      minDate: "today",
      maxDate: new Date().fp_incr(
        <?=isset($settings['max_day_for_service']) ? $settings['max_day_for_service'] : 30?>),
      locale: {
        firstDayOfWeek: 1,
        weekdays: {
          longhand: ['Pazar', 'Pazartesi', 'Salı', 'Çarşamba', 'Perşembe', 'Cuma',
            'Cumartesi'
          ],
          shorthand: ['Paz', 'Pzt', 'Sal', 'Çar', 'Per', 'Cum', 'Cmt']
        },
        months: {
          longhand: ['Ocak', 'Şubat', 'Mart', 'Nisan', 'Mayıs', 'Haziran', 'Temmuz',
            'Ağustos', 'Eylül', 'Ekim', 'Kasım', 'Aralık'
          ],
          shorthand: ['Oca', 'Şub', 'Mar', 'Nis', 'May', 'Haz', 'Tem', 'Ağu', 'Eyl', 'Eki',
            'Kas', 'Ara'
          ]
        },
        today: 'Bugün',
        clear: 'Temizle'
      },
      altFormat: 'd-m-YTH:i',
      onReady: function(selectedDates, dateStr, instance) {
        if (instance.isMobile) {
          $(instance.mobileInput).attr('step', null);
        }
      }
    });
  }
});
</script>
<?php
            } else {
                ?>
<script type="text/javascript">
$(document).ready(function() {
  var startDate = $('#start-date');
  if (startDate.length) {
    var start = startDate.flatpickr({
      minDate: "today",
      maxDate: new Date().fp_incr(
        <?=isset($settings['max_day_for_service']) ? $settings['max_day_for_service'] : 30?>),
      locale: {
        firstDayOfWeek: 1,
        weekdays: {
          longhand: ['Pazar', 'Pazartesi', 'Salı', 'Çarşamba', 'Perşembe', 'Cuma', 'Cumartesi'],
          shorthand: ['Paz', 'Pzt', 'Sal', 'Çar', 'Per', 'Cum', 'Cmt']
        },
        months: {
          longhand: ['Ocak', 'Şubat', 'Mart', 'Nisan', 'Mayıs', 'Haziran', 'Temmuz', 'Ağustos', 'Eylül', 'Ekim',
            'Kasım', 'Aralık'
          ],
          shorthand: ['Oca', 'Şub', 'Mar', 'Nis', 'May', 'Haz', 'Tem', 'Ağu', 'Eyl', 'Eki', 'Kas', 'Ara']
        },
        today: 'Bugün',
        clear: 'Temizle'
      },
      altFormat: 'd-m-YTH:i',
      onReady: function(selectedDates, dateStr, instance) {
        if (instance.isMobile) {
          $(instance.mobileInput).attr('step', null);
        }
      }
    });
  }
});
</script>
<?php
            }
        }
        
    } catch (Exception $e) {
        logRandevu("HATA: " . $e->getMessage());
        echo '<div class="alert alert-danger">Bir hata oluştu: ' . $e->getMessage() . '</div>';
    }
}