<?php
/**
 * JavaScript Syntax Düzeltme Aracı
 * Bu araç, PHP dosyalarındaki JavaScript bölümlerinde yaygın sözdizimi hatalarını düzeltir.
 */

// Çıktıyı HTML olarak biçimlendir
header('Content-Type: text/html; charset=utf-8');

// Ana PHP sayfasını al (index.php veya başka)
$target_file = isset($_GET['file']) ? $_GET['file'] : '/index.php';
$full_path = $_SERVER['DOCUMENT_ROOT'] . $target_file;

if (!file_exists($full_path)) {
    die("Dosya bulunamadı: $full_path");
}

// Dosyayı oku
$content = file_get_contents($full_path);
if ($content === false) {
    die("Dosya okunamadı: $full_path");
}

// JavaScript bölümlerini bul
$js_sections = [];
preg_match_all('/<script[^>]*>(.*?)<\/script>/s', $content, $matches, PREG_OFFSET_CAPTURE);

$fixed_content = $content;
$offset_adjustment = 0;
$fixes = [];

// Her JavaScript bölümünü işle
foreach ($matches[1] as $i => $match) {
    $js_code = $match[0];
    $original_position = $match[1];
    
    // JavaScript kodunu düzelt
    $fixed_js = fix_js_syntax($js_code);
    
    // Değişiklik var mı kontrol et
    if ($fixed_js !== $js_code) {
        // Değişiklikleri kaydet
        $fixes[] = [
            'original' => $js_code,
            'fixed' => $fixed_js,
            'position' => $original_position,
            'script_number' => $i + 1
        ];
        
        // Orijinal içerikte güncelleme yap
        $adjusted_position = $original_position + $offset_adjustment;
        $fixed_content = substr_replace($fixed_content, $fixed_js, $adjusted_position, strlen($js_code));
        $offset_adjustment += strlen($fixed_js) - strlen($js_code);
    }
}

// JavaScript sözdizimi düzeltme fonksiyonu
function fix_js_syntax($js_code) {
    // Sorunlu desenleri düzelt
    
    // 1. Eksik parantezler ve noktalı virgüller
    $js_code = preg_replace('/}\s*else\s*{/', '} else {', $js_code);
    $js_code = preg_replace('/if\s*\(([^)]*)\)\s*(\w)/', 'if ($1) $2', $js_code);
    
    // 2. Sondaki virgülleri temizle
    $js_code = preg_replace('/,\s*}/', '}', $js_code);
    $js_code = preg_replace('/,\s*]/', ']', $js_code);
    
    // 3. Eksik noktalı virgülleri ekle
    $js_code = preg_replace('/}\s*$/', '};', $js_code);
    
    // 4. Tırnak işaretleri içinde kaçışsız tırnak işaretlerini düzelt
    $js_code = preg_replace_callback('/"([^"\\\\]*(?:\\\\.[^"\\\\]*)*)"/', function($m) {
        return '"' . str_replace('"', '\\"', $m[1]) . '"';
    }, $js_code);
    
    // 5. Açılan ama kapanmayan parantezleri düzelt
    $open_parentheses = substr_count($js_code, '(');
    $close_parentheses = substr_count($js_code, ')');
    if ($open_parentheses > $close_parentheses) {
        $js_code .= str_repeat(')', $open_parentheses - $close_parentheses);
    }
    
    // 6. Açılan ama kapanmayan süslü parantezleri düzelt
    $open_braces = substr_count($js_code, '{');
    $close_braces = substr_count($js_code, '}');
    if ($open_braces > $close_braces) {
        $js_code .= str_repeat('}', $open_braces - $close_braces);
    }
    
    // 7. Açılan ama kapanmayan köşeli parantezleri düzelt
    $open_brackets = substr_count($js_code, '[');
    $close_brackets = substr_count($js_code, ']');
    if ($open_brackets > $close_brackets) {
        $js_code .= str_repeat(']', $open_brackets - $close_brackets);
    }
    
    return $js_code;
}

// Düzeltme sonuçlarını göster
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>JavaScript Sözdizimi Düzeltme Aracı</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            margin: 0;
            padding: 20px;
            background-color: #f9f9f9;
            color: #333;
        }
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: #fff;
            padding: 20px;
            border-radius: 5px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        h1 {
            color: #2c3e50;
            margin-top: 0;
            padding-bottom: 10px;
            border-bottom: 2px solid #ecf0f1;
        }
        h2 {
            color: #3498db;
            margin-top: 20px;
        }
        .status {
            margin: 15px 0;
            padding: 10px;
            border-radius: 4px;
        }
        .success {
            background-color: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        .warning {
            background-color: #fff3cd;
            color: #856404;
            border: 1px solid #ffeeba;
        }
        .error {
            background-color: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        .info {
            background-color: #d1ecf1;
            color: #0c5460;
            border: 1px solid #bee5eb;
        }
        pre {
            background: #f8f8f8;
            padding: 10px;
            border-radius: 3px;
            font-family: Consolas, Monaco, 'Andale Mono', monospace;
            font-size: 0.9em;
            overflow-x: auto;
            max-height: 300px;
            border: 1px solid #ddd;
        }
        .diff {
            display: flex;
            flex-wrap: wrap;
            margin-bottom: 20px;
        }
        .diff .original, .diff .fixed {
            flex: 1;
            min-width: 300px;
            margin: 5px;
        }
        .diff h3 {
            margin-top: 0;
            color: #555;
        }
        button {
            background-color: #3498db;
            color: white;
            border: none;
            padding: 10px 15px;
            border-radius: 4px;
            cursor: pointer;
            font-size: 14px;
            margin-top: 10px;
        }
        button:hover {
            background-color: #2980b9;
        }
        .footer {
            margin-top: 30px;
            font-size: 0.9em;
            color: #7f8c8d;
            text-align: center;
        }
        .form-group {
            margin-bottom: 15px;
        }
        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
        }
        .form-group input {
            width: 100%;
            padding: 8px;
            border: 1px solid #ddd;
            border-radius: 4px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>JavaScript Sözdizimi Düzeltme Aracı</h1>
        
        <div class="form-group">
            <form action="" method="get">
                <label for="file">Dosya Yolu (kök dizine göre)</label>
                <input type="text" id="file" name="file" value="<?= htmlspecialchars($target_file) ?>" placeholder="/index.php">
                <button type="submit">Dosyayı Tara</button>
            </form>
        </div>
        
        <div class="status info">
            <strong>Taranan Dosya:</strong> <?= htmlspecialchars($full_path) ?><br>
            <strong>Toplam Script Bölümü:</strong> <?= count($matches[1]) ?>
        </div>
        
        <?php if (empty($fixes)): ?>
        <div class="status success">
            <strong>Sonuç:</strong> Herhangi bir JavaScript sözdizimi hatası bulunamadı veya düzeltilecek bir sorun yok.
        </div>
        <?php else: ?>
        <div class="status warning">
            <strong>Sonuç:</strong> <?= count($fixes) ?> JavaScript bölümünde sözdizimi düzeltmesi yapıldı.
        </div>
        
        <h2>Yapılan Düzeltmeler</h2>
        
        <?php foreach ($fixes as $i => $fix): ?>
        <div class="status info">
            <strong>Script #<?= $fix['script_number'] ?></strong> - Pozisyon: <?= $fix['position'] ?>
        </div>
        <div class="diff">
            <div class="original">
                <h3>Orijinal Kod</h3>
                <pre><?= htmlspecialchars($fix['original']) ?></pre>
            </div>
            <div class="fixed">
                <h3>Düzeltilmiş Kod</h3>
                <pre><?= htmlspecialchars($fix['fixed']) ?></pre>
            </div>
        </div>
        <?php endforeach; ?>
        
        <form action="" method="post">
            <input type="hidden" name="apply_fixes" value="1">
            <input type="hidden" name="file" value="<?= htmlspecialchars($target_file) ?>">
            <button type="submit">Düzeltmeleri Uygula</button>
        </form>
        
        <?php endif; ?>
        
        <?php
        // Düzeltmeleri uygula
        if (isset($_POST['apply_fixes']) && $_POST['apply_fixes'] == 1) {
            $file_to_fix = isset($_POST['file']) ? $_POST['file'] : '/index.php';
            $full_path_to_fix = $_SERVER['DOCUMENT_ROOT'] . $file_to_fix;
            
            if (file_exists($full_path_to_fix) && is_writable($full_path_to_fix)) {
                // Düzeltilmiş içeriği kaydet
                $backup_path = $full_path_to_fix . '.bak_' . date('YmdHis');
                
                // Önce yedek al
                copy($full_path_to_fix, $backup_path);
                
                // Düzeltilmiş dosyayı kaydet
                if (file_put_contents($full_path_to_fix, $fixed_content)) {
                    echo '<div class="status success">';
                    echo '<strong>Başarılı!</strong> Düzeltmeler uygulandı ve dosya kaydedildi.';
                    echo '<br>Yedek dosya oluşturuldu: ' . basename($backup_path);
                    echo '</div>';
                } else {
                    echo '<div class="status error">';
                    echo '<strong>Hata!</strong> Dosya kaydedilemedi.';
                    echo '</div>';
                }
            } else {
                echo '<div class="status error">';
                echo '<strong>Hata!</strong> Dosya bulunamadı veya yazma izni yok: ' . htmlspecialchars($full_path_to_fix);
                echo '</div>';
            }
        }
        ?>
        
        <div class="footer">
            &copy; <?= date('Y') ?> Randevu Sistemi - JavaScript Sözdizimi Düzeltme Aracı v1.0
        </div>
    </div>
</body>
</html>