<?php
/**
 * Ayar Dosyaları Düzeltme Aracı
 * Bu araç ile admin/system/settings klasöründeki dosyalardaki
 * yaygın PHP hataları ve güvenlik açıkları düzeltilir.
 */

// Çıktıyı HTML olarak biçimlendir
header('Content-Type: text/html; charset=utf-8');
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Ayar Dosyaları Düzeltme Aracı</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            margin: 0;
            padding: 20px;
            background-color: #f9f9f9;
            color: #333;
        }
        .container {
            max-width: 900px;
            margin: 0 auto;
            background: #fff;
            padding: 20px;
            border-radius: 5px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        h1 {
            color: #2c3e50;
            margin-top: 0;
            padding-bottom: 10px;
            border-bottom: 2px solid #ecf0f1;
        }
        h2 {
            color: #3498db;
            margin-top: 20px;
        }
        .status {
            margin: 15px 0;
            padding: 10px;
            border-radius: 4px;
        }
        .success {
            background-color: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        .warning {
            background-color: #fff3cd;
            color: #856404;
            border: 1px solid #ffeeba;
        }
        .error {
            background-color: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        .info {
            background-color: #d1ecf1;
            color: #0c5460;
            border: 1px solid #bee5eb;
        }
        ul {
            list-style-type: none;
            padding-left: 10px;
        }
        ul li {
            margin-bottom: 8px;
            padding-left: 20px;
            position: relative;
        }
        ul li:before {
            content: "✓";
            position: absolute;
            left: 0;
            color: #27ae60;
        }
        code {
            background: #f8f8f8;
            padding: 2px 4px;
            border-radius: 3px;
            font-family: Consolas, Monaco, 'Andale Mono', monospace;
            font-size: 0.9em;
            color: #e74c3c;
        }
        .footer {
            margin-top: 30px;
            font-size: 0.9em;
            color: #7f8c8d;
            text-align: center;
        }
        button {
            background-color: #3498db;
            color: white;
            border: none;
            padding: 10px 15px;
            border-radius: 4px;
            cursor: pointer;
            font-size: 14px;
            margin-top: 10px;
        }
        button:hover {
            background-color: #2980b9;
        }
        button:disabled {
            background-color: #95a5a6;
            cursor: not-allowed;
        }
        .details {
            margin-top: 5px;
            font-size: 0.9em;
            color: #555;
        }
        .section {
            margin-bottom: 20px;
            padding-bottom: 20px;
            border-bottom: 1px solid #eee;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>Ayar Dosyaları Düzeltme Aracı</h1>
        
        <div class="section">
            <h2>1. Durumu Kontrol Ediliyor...</h2>
            
            <?php
            // Root dizini tanımla
            $root_dir = $_SERVER['DOCUMENT_ROOT'];
            $settings_dir = $root_dir . '/admin/system/settings';
            
            // Dizin var mı kontrol et
            if (!file_exists($settings_dir) || !is_dir($settings_dir)) {
                echo '<div class="status error">Ayar dizini bulunamadı: ' . $settings_dir . '</div>';
                exit;
            }
            
            echo '<div class="status info">Ayar dizini bulundu: ' . $settings_dir . '</div>';
            
            // Dosyaları listele
            $php_files = glob($settings_dir . '/*.php');
            $total_files = count($php_files);
            
            echo '<div class="status info">Toplam ' . $total_files . ' PHP dosyası bulundu.</div>';
            
            // Sorunlu alanları tanımla
            $problems = [
                'session_start' => [
                    'pattern' => '/(\s*)session_start\(\);/',
                    'replacement' => "$1// Oturumu başlat eğer başlamamışsa\n$1if (session_status() == PHP_SESSION_NONE) {\n$1    session_start();\n$1}",
                    'description' => 'session_start() fonksiyonu session_status() kontrolüyle düzeltildi',
                    'files' => []
                ],
                'trim_null' => [
                    'pattern' => '/trim\((\$[^)]+)\)/',
                    'replacement' => '(!empty($1) ? trim($1) : "")',
                    'description' => 'trim() fonksiyonu null kontrolüyle düzeltildi',
                    'files' => []
                ],
                'explode_null' => [
                    'pattern' => '/explode\(([^,]+),\s*(\$[^)]+)\)/',
                    'replacement' => '(!empty($2) ? explode($1, $2) : [])',
                    'description' => 'explode() fonksiyonu null kontrolüyle düzeltildi',
                    'files' => []
                ],
                'error_suppress' => [
                    'pattern' => '/@(trim|explode|implode|strpos|strstr|substr|str_replace)/',
                    'replacement' => '$1',
                    'description' => '@ işareti ile hata bastırması kaldırıldı',
                    'files' => []
                ]
            ];
            
            // Düzeltme yapılacak dosyaları tara
            $fixed_files = [];
            $error_files = [];
            
            foreach ($php_files as $file) {
                $filename = basename($file);
                $content = file_get_contents($file);
                
                if ($content === false) {
                    $error_files[] = $filename;
                    continue;
                }
                
                $needs_fixing = false;
                
                foreach ($problems as $problem => $data) {
                    if (preg_match($data['pattern'], $content)) {
                        $problems[$problem]['files'][] = $filename;
                        $needs_fixing = true;
                    }
                }
                
                if ($needs_fixing) {
                    $fixed_files[] = $filename;
                }
            }
            
            // Sonuçları göster
            echo '<h2>2. Bulunan Sorunlar</h2>';
            
            $has_problems = false;
            
            foreach ($problems as $problem => $data) {
                if (!empty($data['files'])) {
                    $has_problems = true;
                    echo '<div class="status warning">';
                    echo '<strong>' . $data['description'] . '</strong> sorununu içeren dosyalar:';
                    echo '<ul>';
                    foreach ($data['files'] as $file) {
                        echo '<li>' . $file . '</li>';
                    }
                    echo '</ul></div>';
                }
            }
            
            if (!$has_problems) {
                echo '<div class="status success">Hiçbir dosyada bilinen sorun bulunamadı.</div>';
            }
            
            // Düzeltilecek dosya sayısını kontrol et
            if (!empty($fixed_files)) {
                echo '<h2>3. Düzeltilecek Dosyalar</h2>';
                echo '<div class="status info">';
                echo 'Toplam ' . count($fixed_files) . ' dosyada düzeltme yapılacak:';
                echo '<ul>';
                foreach ($fixed_files as $file) {
                    echo '<li>' . $file . '</li>';
                }
                echo '</ul></div>';
                
                // Düzeltme butonunu göster
                echo '<form method="post">';
                echo '<input type="hidden" name="fix_action" value="fix_files">';
                echo '<button type="submit">Dosyaları Düzelt</button>';
                echo '</form>';
            }
            
            // Hata dosyalarını göster
            if (!empty($error_files)) {
                echo '<div class="status error">';
                echo 'Toplam ' . count($error_files) . ' dosya okunamadı:';
                echo '<ul>';
                foreach ($error_files as $file) {
                    echo '<li>' . $file . '</li>';
                }
                echo '</ul></div>';
            }
            
            // Eğer düzeltme isteği geldiyse işle
            if (isset($_POST['fix_action']) && $_POST['fix_action'] == 'fix_files') {
                echo '<h2>4. Düzeltme Sonuçları</h2>';
                
                $success_count = 0;
                $error_count = 0;
                $fixed_details = [];
                
                foreach ($fixed_files as $filename) {
                    $filepath = $settings_dir . '/' . $filename;
                    $content = file_get_contents($filepath);
                    
                    if ($content === false) {
                        $error_count++;
                        echo '<div class="status error">' . $filename . ': Dosya okunamadı.</div>';
                        continue;
                    }
                    
                    $original_content = $content;
                    $fixed_details[$filename] = [];
                    
                    // Tüm sorunları düzelt
                    foreach ($problems as $problem => $data) {
                        $count = 0;
                        $content = preg_replace($data['pattern'], $data['replacement'], $content, -1, $count);
                        
                        if ($count > 0) {
                            $fixed_details[$filename][] = $data['description'] . ' (' . $count . ' yer)';
                        }
                    }
                    
                    // Dosya değiştiyse kaydet
                    if ($content !== $original_content) {
                        if (file_put_contents($filepath, $content)) {
                            $success_count++;
                        } else {
                            $error_count++;
                            echo '<div class="status error">' . $filename . ': Dosya yazılamadı.</div>';
                        }
                    }
                }
                
                // Sonuçları göster
                if ($success_count > 0) {
                    echo '<div class="status success">';
                    echo 'Toplam ' . $success_count . ' dosya başarıyla düzeltildi.';
                    echo '<ul>';
                    foreach ($fixed_details as $filename => $details) {
                        if (!empty($details)) {
                            echo '<li>' . $filename . ' <div class="details">' . implode(', ', $details) . '</div></li>';
                        }
                    }
                    echo '</ul></div>';
                }
                
                if ($error_count > 0) {
                    echo '<div class="status error">';
                    echo 'Toplam ' . $error_count . ' dosya düzeltilemedi.';
                    echo '</div>';
                }
            }
            ?>
        </div>
        
        <div class="footer">
            &copy; <?php echo date('Y'); ?> Randevu Sistemi - Ayar Dosyaları Düzeltme Aracı v1.0
        </div>
    </div>
</body>
</html>