// Randevu System Service Worker
// Version: 3.3.5

const CACHE_NAME = "randevu-cache-v3.3.5";
const OFFLINE_URL = "/index.php";

// İlk yüklemede önbelleğe alınacak dosyalar
const PRECACHE_URLS = [
  "/",
  "/index.php",
  "/manifest.json",
  "/assets/css/style.css",
  "/assets/js/script.js",
  "/assets/js/jquery-3.6.0.min.js",
  "/assets/css/bootstrap.min.css",
  "/assets/js/bootstrap.min.js",
  "/assets/css/animate.min.css",
  "/assets/js/intlTelInput.js",
  "/assets/css/intlTelInput.css",
  "/assets/images/flags/flags.png",
  "/assets/images/flags/flags@2x.png",
  "/pwa.png",
  "/assets/images/logo/logo.png",
  "/styles.css",
  "/app.js",
  "https://cdnjs.cloudflare.com/ajax/libs/intl-tel-input/17.0.13/js/utils.js",
];

// Service Worker kurulumunda önbelleğe alma - her URL'yi tek tek işle
self.addEventListener("install", (event) => {
  console.log("[ServiceWorker] Yükleniyor");

  event.waitUntil(
    caches
      .open(CACHE_NAME)
      .then((cache) => {
        console.log("[ServiceWorker] Önbellek açıldı");

        // Her URL'yi tek tek önbelleğe al, hata olursa işlemi durdurma
        return Promise.all(
          PRECACHE_URLS.map((url) => {
            return fetch(url)
              .then((response) => {
                if (!response.ok) {
                  throw new Error("Yanıt başarısız: " + response.status);
                }
                return cache.put(url, response);
              })
              .catch((error) => {
                console.warn(
                  "[ServiceWorker] URL önbelleğe alınamadı:",
                  url,
                  error
                );
                // Hata olsa bile işleme devam et
                return Promise.resolve();
              });
          })
        );
      })
      .then(() => {
        console.log("[ServiceWorker] Tüm kaynaklar önbelleğe alındı");
        return self.skipWaiting();
      })
      .catch((error) => {
        console.error("[ServiceWorker] Önbelleğe alma hatası:", error);
      })
  );
});

// Yeni Service Worker etkinleştirildiğinde eski önbellekleri temizleme
self.addEventListener("activate", (event) => {
  console.log("[ServiceWorker] Etkinleştiriliyor");

  event.waitUntil(
    caches
      .keys()
      .then((cacheNames) => {
        return Promise.all(
          cacheNames
            .filter((cacheName) => {
              return (
                cacheName.startsWith("randevu-cache-") &&
                cacheName !== CACHE_NAME
              );
            })
            .map((cacheName) => {
              console.log(
                "[ServiceWorker] Eski önbellek siliniyor:",
                cacheName
              );
              return caches.delete(cacheName);
            })
        );
      })
      .then(() => {
        console.log("[ServiceWorker] Etkinleştirildi ve kontrol sağlandı");
        return self.clients.claim();
      })
  );
});

// Ağ isteklerini yakalama ve önbellekten sunma
self.addEventListener("fetch", (event) => {
  // API isteklerini ve AJAX'ı önbellekleme
  if (
    event.request.url.includes("/admin/system/ajax/") ||
    event.request.url.includes("/admin/system/settings/")
  ) {
    // API isteklerini normal ağ üzerinden geçir
    return;
  }

  // CORS sorunlarını önlemek için - CDN isteklerini doğrudan geçir
  if (
    event.request.url.includes("cdn.") ||
    event.request.url.includes("cdnjs.") ||
    event.request.url.includes("jsdelivr.net") ||
    event.request.url.includes("code.jquery.com") ||
    event.request.url.includes("stackpath.bootstrapcdn.com") ||
    event.request.url.includes("datatables.net") ||
    !event.request.url.includes(self.location.origin)
  ) {
    // CDN isteklerini doğrudan geçir, service worker'ın müdahale etmesini engelle
    return;
  }

  // POST isteklerini önbellekleme
  if (event.request.method !== 'GET') {
    // POST isteklerini normal ağdan geçir ve önbellekleme
    return;
  }

  // GET istekleri için cache-first stratejisi
  event.respondWith(
    caches.match(event.request).then((response) => {
      if (response) {
        // Önbellekte bulundu
        return response;
      }

      // Önbellekte yoksa, ağdan al
      return fetch(event.request)
        .then((response) => {
          // Geçerli bir yanıt alındıysa önbelleğe ekle
          if (response && response.status === 200) {
            const responseToCache = response.clone();
            caches.open(CACHE_NAME).then((cache) => {
              // PUT işlemi sadece GET yanıtlarında çalışacak
              cache.put(event.request, responseToCache);
            });
          }
          return response;
        })
        .catch((error) => {
          console.error("Fetch hatası:", error);
          // Çevrimdışıysa ve HTML isteği ise, offline sayfasını göster
          if (event.request.mode === "navigate") {
            return caches.match(OFFLINE_URL);
          }
          // Diğer durumlar için boş yanıt
          return new Response();
        });
    })
  );
});

// PWA ile ilgili mesajları işleme
self.addEventListener("message", (event) => {
  if (event.data && event.data.type === "SKIP_WAITING") {
    self.skipWaiting();
  }
});
