// ==================================================
// Randevu - Modern Appointment Booking System
// Version: 3.1.9
// ==================================================

// Global variables for better form state tracking
var currentTab = 0; // Global variable to track current form step

// Wait for document to be fully loaded
window.addEventListener('DOMContentLoaded', function() {
  console.log('DOM fully loaded, initializing form...');
  
  // Initialize form state
  showTab(currentTab);
  setupFormListeners();
  
  // Load translations
  initTranslations();
  
  // Setup PWA prompt
  setupPWAInstall();
  
  // Mobil cihazlar için özel ayarlamalar
  if (window.innerWidth < 768) {
    setupMobileOptimizations();
  }
  
  console.log('Form initialization complete');
});

// Setup all form-related event listeners
function setupFormListeners() {
  // Step navigation buttons - forward
  var nextButtons = document.querySelectorAll('.next-step');
  for (var i = 0; i < nextButtons.length; i++) {
    nextButtons[i].addEventListener('click', function(e) {
      e.preventDefault();
      console.log('Next button clicked, current tab:', currentTab);
      nextStep();
    });
  }
  
  // Step navigation buttons - backward
  var prevButtons = document.querySelectorAll('.prev-step');
  for (var i = 0; i < prevButtons.length; i++) {
    prevButtons[i].addEventListener('click', function(e) {
      e.preventDefault();
      console.log('Previous button clicked, current tab:', currentTab);
      prevStep();
    });
  }
  
  // Form submission button
  var submitButton = document.getElementById('randevuEkleBtn');
  if (submitButton) {
    submitButton.addEventListener('click', function(e) {
      e.preventDefault();
      console.log('Submit button clicked');
      submitForm();
    });
  } else {
    console.error('Submit button (randevuEkleBtn) not found!');
  }
  
  // Initialize custom option selectors
  initializeCustomSelects();
}

// Show the specified tab and update UI accordingly
function showTab(n) {
  console.log('Showing tab', n);
  
  // Get all form panels
  var formPanels = document.getElementsByClassName("multisteps_form_panel");
  if (formPanels.length === 0) {
    console.error('No form panels found!');
    return;
  }
  
  // Ensure tab index is valid
  if (n >= formPanels.length) {
    console.error('Tab index out of bounds:', n, 'max:', formPanels.length - 1);
    return;
  }
  
  // Hide all panels
  for (var i = 0; i < formPanels.length; i++) {
    formPanels[i].style.display = "none";
  }
  
  // Show the current panel
  formPanels[n].style.display = "block";
  
  // Update progress indicators
  updateProgressSteps(n);
  updateProgressBar(n, formPanels.length);
  
  console.log('Tab', n, 'is now visible');
}

// Move to next step if validation passes
function nextStep() {
  console.log('Attempting to move to next step from current tab:', currentTab);
  
  // Validate current step before proceeding
  if (!validateCurrentStep()) {
    console.warn('Validation failed for tab', currentTab);
    return false;
  }
  
  // Get form panels
  var formPanels = document.getElementsByClassName("multisteps_form_panel");
  
  // Hide current panel
  formPanels[currentTab].style.display = "none";
  
  // Increment tab counter
  currentTab++;
  console.log('Moving to tab', currentTab);
  
  // If we've reached the last step, prepare summary
  if (currentTab === formPanels.length - 1) {
    updateSummary();
  }
  
  // Show the next tab
  if (currentTab < formPanels.length) {
    showTab(currentTab);
  }
  
  return true;
}

// Move to previous step
function prevStep() {
  // Get form panels
  var formPanels = document.getElementsByClassName("multisteps_form_panel");
  
  // Make sure we're not on the first step
  if (currentTab > 0) {
    // Hide current panel
    formPanels[currentTab].style.display = "none";
    
    // Decrement tab counter
    currentTab--;
    console.log('Moving back to tab', currentTab);
    
    // Show the previous tab
    showTab(currentTab);
  }
}

// Update progress steps indicators
function updateProgressSteps(currentStepIndex) {
  var steps = document.querySelectorAll('.progress-step');
  
  if (steps.length === 0) {
    console.warn('No progress steps found to update');
    return;
  }
  
  // Update each step's status
  steps.forEach(function(step, index) {
    // Remove all status classes
    step.classList.remove('active', 'complete');
    
    // Mark appropriate status
    if (index < currentStepIndex) {
      step.classList.add('complete');
    } else if (index === currentStepIndex) {
      step.classList.add('active');
    }
  });
}

// Update progress bar width based on current step
function updateProgressBar(currentStep, totalSteps) {
  var progressBar = document.querySelector('.progress-bar');
  
  if (!progressBar) {
    console.warn('Progress bar element not found');
    return;
  }
  
  var progressPercentage = (currentStep / (totalSteps - 1)) * 100;
  progressBar.style.width = progressPercentage + '%';
}

// Validate the current step's inputs
function validateCurrentStep() {
  console.log('Validating step', currentTab);
  var isValid = true;
  
  // Step 1: Personal Information validation
  if (currentTab === 0) {
    var nameField = document.getElementById('musteriAd');
    var phoneField = document.getElementById('musteriTelefon');
    
    if (!nameField || !phoneField) {
      console.error('Required form fields not found');
      return false;
    }
    
    // Validate name
    if (!nameField.value.trim()) {
      showError('Hata', 'Lütfen adınızı giriniz');
      nameField.classList.add('is-invalid');
      isValid = false;
    } else {
      nameField.classList.remove('is-invalid');
      
      // Additional validation for name format (first + last name)
      if (nameField.value.trim().split(' ').length < 2) {
        showError('Hata', 'Lütfen ad ve soyadınızı giriniz');
        nameField.classList.add('is-invalid');
        isValid = false;
      }
    }
    
    // Validate phone
    if (!phoneField.value.trim()) {
      showError('Hata', 'Lütfen telefon numaranızı giriniz');
      phoneField.classList.add('is-invalid');
      isValid = false;
    } else {
      phoneField.classList.remove('is-invalid');
    }
  }
  
  // Step 2: Service selection validation
  else if (currentTab === 1) {
    var serviceField = document.getElementById('service');
    var providerField = document.getElementById('providerID');
    
    // Validate service
    if (!serviceField || !serviceField.value || serviceField.value === '') {
      showError('Hata', 'Lütfen hizmet seçiniz');
      isValid = false;
    }
    
    // Validate provider
    if (!providerField || !providerField.value || providerField.value === '') {
      showError('Hata', 'Lütfen personel seçiniz');
      isValid = false;
    }
  }
  
  // Step 3: Date and time validation
  else if (currentTab === 2) {
    var dateField = document.getElementById('start-date');
    var timeSelected = document.querySelector(".custom-option-item-check:checked");
    
    // Validate date
    if (!dateField || !dateField.value || dateField.value === '') {
      showError('Hata', 'Lütfen tarih seçiniz');
      isValid = false;
    }
    
    // Validate time
    if (!timeSelected) {
      showError('Hata', 'Lütfen saat seçiniz');
      isValid = false;
    }
  }
  
  console.log('Validation result:', isValid);
  return isValid;
}

// Initialize custom select/option items
function initializeCustomSelects() {
  // For providers and time slots selection - using event delegation
  document.addEventListener('click', function(e) {
    // Time slot selection
    if (e.target.closest('.time-slot')) {
      var clickedSlot = e.target.closest('.time-slot');
      var container = clickedSlot.closest('.time-slots-container');
      
      if (container) {
        // Remove selected class from all slots
        var timeSlots = container.querySelectorAll('.time-slot');
        timeSlots.forEach(function(slot) {
          slot.classList.remove('selected');
        });
        
        // Add selected class to clicked slot
        clickedSlot.classList.add('selected');
        
        // Find and check the radio button
        var radio = clickedSlot.querySelector('input[type="radio"]');
        if (radio) radio.checked = true;
      }
    }
    
    // Provider selection
    if (e.target.closest('.custom-option-item')) {
      var clickedOption = e.target.closest('.custom-option-item');
      var container = clickedOption.closest('.custom-options');
      
      if (container) {
        // Remove selected class from all options
        var options = container.querySelectorAll('.custom-option-item');
        options.forEach(function(option) {
          option.classList.remove('selected');
        });
        
        // Add selected class to clicked option
        clickedOption.classList.add('selected');
        
        // Find and check the radio button
        var radio = clickedOption.querySelector('input[type="radio"]');
        if (radio) radio.checked = true;
      }
    }
  });
}

// Update summary information before showing last step
function updateSummary() {
  console.log('Updating summary data for final step');
  
  // Personal information
  var adSoyadElement = document.getElementById('OzetAdSoyad');
  var telefonElement = document.getElementById('OzetTelefon');
  
  if (adSoyadElement && telefonElement) {
    var musteriAd = document.getElementById('musteriAd');
    var musteriTelefon = document.getElementById('musteriTelefon');
    
    if (musteriAd) adSoyadElement.textContent = musteriAd.value;
    if (musteriTelefon) telefonElement.textContent = musteriTelefon.value;
  }
  
  // Service information
  var hizmetElement = document.getElementById('OzetHizmetAd');
  var calisanElement = document.getElementById('OzetCalisanAd');
  
  if (hizmetElement) {
    var serviceSelect = document.getElementById('service');
    if (serviceSelect && serviceSelect.selectedIndex >= 0) {
      hizmetElement.textContent = serviceSelect.options[serviceSelect.selectedIndex].textContent;
    }
  }
  
  if (calisanElement) {
    var calisanAd = document.getElementById('providerAdSoyad');
    if (calisanAd) calisanElement.textContent = calisanAd.value;
  }
  
  // Date and time information
  var tarihElement = document.getElementById('OzetTarih');
  var saatElement = document.getElementById('OzetSaat');
  
  if (tarihElement) {
    var dateField = document.getElementById('start-date');
    if (dateField) tarihElement.textContent = dateField.value;
  }
  
  if (saatElement) {
    var selectedTimeSlot = document.querySelector(".custom-option-item-check:checked");
    if (selectedTimeSlot) saatElement.textContent = selectedTimeSlot.value;
  }
}

// Show error message using SweetAlert2
function showError(title, message) {
  if (typeof Swal !== 'undefined') {
    Swal.fire({
      title: title,
      text: message,
      icon: 'error',
      confirmButtonColor: '#4361ee',
      confirmButtonText: 'Tamam'
    });
  } else {
    // Fallback if SweetAlert not loaded
    alert(title + ': ' + message);
  }
}

// Show loading indicator
function showLoading(message) {
  if (typeof Swal !== 'undefined') {
    Swal.fire({
      title: 'İşleniyor',
      html: message || 'Lütfen bekleyiniz',
      allowOutsideClick: false,
      showConfirmButton: false,
      willOpen: function() {
        Swal.showLoading();
      }
    });
  }
}

// Submit form data
function submitForm() {
  console.log('Starting form submission process');
  
  // Final validation
  if (!validateCurrentStep()) {
    console.warn('Final validation failed, form not submitted');
    return;
  }
  
  // Get the submit button
  var btn = document.getElementById('randevuEkleBtn');
  if (!btn) {
    console.error('Submit button not found!');
    return;
  }
  
  // Show loading state on button
  btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> İşleniyor';
  btn.disabled = true;
  
  // Show loading indicator
  showLoading();
  
  // Get form data
  var form = document.getElementById('wizard');
  var formData = new FormData(form);
  
  // Determine payment method
  var paymentMethodElement = document.getElementById('odemeYontemi');
  var paymentMethod = paymentMethodElement ? paymentMethodElement.value : 'Nakit';
  
  // Submit to appropriate endpoint based on payment method
  if (paymentMethod === 'Kredi') {
    submitPaymentForm(formData, btn);
  } else {
    submitRegularForm(formData, btn);
  }
}

// Submit payment form
function submitPaymentForm(formData, btn) {
  console.log('Submitting payment form');
  
  fetch('/admin/system/settings/paymentCreate.php', {
    method: 'POST',
    body: formData
  })
  .then(function(response) {
    if (!response.ok) {
      throw new Error('Network response was not ok: ' + response.status);
    }
    return response.text();
  })
  .then(function(data) {
    console.log('Payment form response:', data);
    
    if (data === 'varvar') {
      Swal.fire({
        title: 'Üzgünüz',
        text: 'Lütfen başka bir seçim yapınız',
        icon: 'error',
        confirmButtonColor: '#4361ee',
        confirmButtonText: 'Tamam'
      }).then(function() {
        location.reload();
      });
    } else {
      location.href = "payment/" + data;
    }
  })
  .catch(function(error) {
    console.error('Payment submission error:', error);
    
    Swal.fire({
      title: 'Hata',
      text: 'İşlem sırasında bir hata oluştu: ' + error.message,
      icon: 'error',
      confirmButtonColor: '#4361ee',
      confirmButtonText: 'Tamam'
    });
    
    // Reset button state
    btn.innerHTML = 'Oluştur <i class="fas fa-check ms-2"></i>';
    btn.disabled = false;
  });
}

// Submit regular form
function submitRegularForm(formData, btn) {
  console.log('Submitting regular form');
  
  fetch('/admin/system/settings/eral.php', {
    method: 'POST',
    body: formData
  })
  .then(function(response) {
    if (!response.ok) {
      throw new Error('Network response was not ok: ' + response.status);
    }
    return response.text();
  })
  .then(function(data) {
    console.log('Regular form response:', data);
    
    if (data === 'varvar') {
      Swal.fire({
        title: 'Üzgünüz',
        text: 'Lütfen başka bir seçim yapınız',
        icon: 'error',
        confirmButtonColor: '#4361ee',
        confirmButtonText: 'Tamam'
      }).then(function() {
        location.reload();
      });
    } else {
      var paymentMethodElement = document.getElementById('odemeYontemi');
      var paymentMethod = paymentMethodElement ? paymentMethodElement.value : 'Nakit';
      
      if (paymentMethod === 'Havale') {
        location.href = "hsuccess";
      } else {
        location.href = "success";
      }
    }
  })
  .catch(function(error) {
    console.error('Regular form submission error:', error);
    
    Swal.fire({
      title: 'Hata',
      text: 'İşlem sırasında bir hata oluştu: ' + error.message,
      icon: 'error',
      confirmButtonColor: '#4361ee',
      confirmButtonText: 'Tamam'
    });
    
    // Reset button state
    btn.innerHTML = 'Oluştur <i class="fas fa-check ms-2"></i>';
    btn.disabled = false;
  });
}

// Setup PWA installation
function setupPWAInstall() {
  var deferredPrompt;
  
  // PWA container oluştur
  var pwaContainer = document.createElement('div');
  pwaContainer.className = 'pwa-install-container';
  
  // PWA bildirimi oluştur
  var pwaPrompt = document.createElement('div');
  pwaPrompt.className = 'pwa-install-prompt';
  pwaPrompt.innerHTML = `
    <div class="pwa-prompt-content">
      <div class="pwa-prompt-icon">
        <img src="/assets/192x192.png" alt="App Icon" width="42" height="42">
      </div>
      <div class="pwa-prompt-text">
        <div class="pwa-prompt-title">Randevu Uygulamasını Yükle</div>
        <div class="pwa-prompt-description">Daha hızlı erişim için uygulamayı cihazınıza ekleyin</div>
      </div>
    </div>
    <div class="pwa-prompt-actions">
      <button class="pwa-install-button">Yükle</button>
      <button class="pwa-dismiss-button">Daha Sonra</button>
    </div>
  `;
  
  // Containeri DOM'a ekle
  pwaContainer.appendChild(pwaPrompt);
  document.body.appendChild(pwaContainer);
  
  // Eski yükleme butonunu gizle (eğer varsa)
  var legacyInstallButton = document.getElementById('installButton');
  if (legacyInstallButton) {
    legacyInstallButton.style.display = 'none';
  }
  
  // Yükleme promtunu yakala
  window.addEventListener('beforeinstallprompt', function(e) {
    console.log('beforeinstallprompt olayı tetiklendi');
    
    // Varsayılan davranışı engelle
    e.preventDefault();
    
    // Promptu kaydet
    deferredPrompt = e;
    
    // Bildirim banner'ını hemen göster
    pwaContainer.style.display = 'block';
    
    // Debug log
    console.log('PWA yükleme bildirimi gösterildi');
    
    // Manuel olarak acil göstermek için localStorage'daki kaydı temizle
    localStorage.removeItem('pwa-prompt-dismissed');
  });
  
  // Yükleme butonuna tıklama olayı
  var installButton = document.querySelector('.pwa-install-button');
  if (installButton) {
    installButton.addEventListener('click', async function() {
      if (!deferredPrompt) {
        console.log('Yükleme bildirimi hazır değil - Manuel tarayıcı promptu deneniyor');
        
        // Eğer deferredPrompt yoksa, manuel bir yükleme denemesi yap
        try {
          // Bazı tarayıcılarda location'a based bir yükleme tetiklenebilir
          if (navigator.standalone === undefined && !window.matchMedia('(display-mode: standalone)').matches) {
            Swal.fire({
              title: 'Tarayıcı Menüsünü Kullanın',
              text: 'Uygulamayı yüklemek için tarayıcı menüsünden "Ana Ekrana Ekle" seçeneğini kullanın',
              icon: 'info',
              confirmButtonColor: '#ff950e',
              confirmButtonText: 'Tamam'
            });
          }
        } catch (err) {
          console.error('Manuel yükleme hatası:', err);
        }
        
        return;
      }
      
      // Yükleme promptunu göster
      deferredPrompt.prompt();
      
      // Kullanıcının yanıtını bekle
      try {
        const { outcome } = await deferredPrompt.userChoice;
        console.log('PWA install outcome:', outcome);
      } catch (err) {
        console.error('Yükleme hatası:', err);
      }
      
      // Prompt değişkenini sıfırla
      deferredPrompt = null;
      
      // Yükleme bildirimini gizle
      pwaContainer.style.display = 'none';
    });
  }
  
  // "Daha Sonra" butonuna tıklama olayı
  var dismissButton = document.querySelector('.pwa-dismiss-button');
  if (dismissButton) {
    dismissButton.addEventListener('click', function() {
      pwaContainer.style.display = 'none';
      
      // 1 gün sonra tekrar göster
      localStorage.setItem('pwa-prompt-dismissed', Date.now());
    });
  }
  
  // Uygulama başarıyla yüklendiğinde
  window.addEventListener('appinstalled', function() {
    console.log('PWA başarıyla yüklendi');
    pwaContainer.style.display = 'none';
    
    // Başarı bildirimi göster
    Swal.fire({
      title: 'Uygulama Yüklendi!',
      text: 'Randevu uygulaması başarıyla cihazınıza yüklendi.',
      icon: 'success',
      confirmButtonColor: '#ff950e',
      confirmButtonText: 'Tamam'
    });
  });
  
  // Zaten yüklüyse bildirimi gösterme
  if (window.matchMedia('(display-mode: standalone)').matches || 
      (navigator.standalone && navigator.standalone === true)) {
    console.log('Uygulama zaten yüklü - bildirim gösterilmeyecek');
    pwaContainer.style.display = 'none';
    return;
  }
  
  // Daha önce reddedilmiş mi kontrol et
  var lastDismissed = localStorage.getItem('pwa-prompt-dismissed');
  if (lastDismissed) {
    var hoursSinceDismissed = (Date.now() - parseInt(lastDismissed)) / (1000 * 60 * 60);
    if (hoursSinceDismissed < 24) {
      console.log('Bildirim son 24 saat içinde reddedilmiş - gizleniyor');
      pwaContainer.style.display = 'none';
    } else {
      // 24 saatten fazla geçmişse göster
      console.log('Bildirim 24 saatten fazla zaman önce reddedilmiş - tekrar gösteriliyor');
      pwaContainer.style.display = 'block';
    }
  } else {
    // Hiç reddedilmemişse, hemen göster
    console.log('Bildirim hiç reddedilmemiş - gösteriliyor');
    setTimeout(function() {
      pwaContainer.style.display = 'block';
    }, 3000); // 3 saniye sonra göster
  }
  
  // Manuel olarak PWA yükleme bildirimini zorla göster
  window.showPWAInstallPrompt = function() {
    pwaContainer.style.display = 'block';
    localStorage.removeItem('pwa-prompt-dismissed');
    
    if (!deferredPrompt) {
      console.log('Yükleme bildirimi hazır değil - manuel browser prompt açılamaz');
      return false;
    } else {
      console.log('Yükleme bildirimi gösterildi');
      return true;
    }
  };
  
  // Sayfanın sonunda manuel olarak bildirimi göster
  setTimeout(function() {
    if (pwaContainer.style.display !== 'block') {
      pwaContainer.style.display = 'block';
      console.log('PWA bildirimi sayfa yüklendikten 3 saniye sonra gösterildi');
    }
  }, 3000);
}

// Initialize translations
var translations = {
  error: 'Hata',
  pleasename: 'Lütfen adınızı giriniz',
  pleasephone: 'Lütfen telefon numaranızı giriniz',
  pleaseservice: 'Lütfen hizmet seçiniz',
  pleaseemp: 'Lütfen personel seçiniz',
  pleaseappdate: 'Lütfen tarih seçiniz',
  pleaseapptime: 'Lütfen saat seçiniz',
  okey: 'Tamam',
  sorry: 'Üzgünüz',
  otherselect: 'Lütfen başka bir seçim yapınız',
  processing: 'İşleniyor',
  wait: 'Lütfen bekleyiniz',
  create: 'Oluştur',
  processerror: 'İşlem sırasında bir hata oluştu:'
};

// Load translations from window if available
function initTranslations() {
  if (typeof window.translations !== 'undefined') {
    Object.keys(translations).forEach(function(key) {
      if (window.translations[key]) {
        translations[key] = window.translations[key];
      }
    });
  }
}

// Mobil cihazlar için ek optimizasyonlar - Added in v3.1.9
function setupMobileOptimizations() {
  console.log("Setting up mobile optimizations...");
  
  // Input alanlarına tıklayınca otomatik scroll
  const inputs = document.querySelectorAll('input, select, textarea');
  inputs.forEach(input => {
    input.addEventListener('focus', function() {
      // Ekranın ortasına scroll
      setTimeout(() => {
        const rect = this.getBoundingClientRect();
        const scrollTop = window.pageYOffset || document.documentElement.scrollTop;
        const targetY = rect.top + scrollTop - (window.innerHeight / 2);
        window.scrollTo({top: targetY, behavior: 'smooth'});
      }, 300);
    });
  });
  
  // Mobil cihazlar için form doğrulama mesajlarını geliştir
  document.querySelectorAll('.form-control').forEach(control => {
    control.addEventListener('blur', function() {
      if (this.value.trim() === '' && this.hasAttribute('required')) {
        this.classList.add('is-invalid');
        
        // Hata mesajı DOM'a ekleme
        let errorElementId = this.id + '-error';
        let errorElement = document.getElementById(errorElementId);
        
        if (!errorElement) {
          errorElement = document.createElement('div');
          errorElement.id = errorElementId;
          errorElement.className = 'invalid-feedback';
          this.parentNode.appendChild(errorElement);
        }
        
        errorElement.textContent = 'Bu alan zorunludur';
      } else {
        this.classList.remove('is-invalid');
        let errorElement = document.getElementById(this.id + '-error');
        if (errorElement) errorElement.remove();
      }
    });
  });
}

// Form doğrulama fonksiyonu geliştirildi - Updated in v3.1.9
function validateCurrentStep() {
  console.log('Validating step', currentTab);
  var isValid = true;
  
  // Step 1: Kişisel bilgiler doğrulama
  if (currentTab === 0) {
    var nameField = document.getElementById('musteriAd');
    var phoneField = document.getElementById('musteriTelefon');
    
    if (!nameField || !phoneField) {
      console.error('Gerekli form alanları bulunamadı');
      return false;
    }
    
    // İsim doğrulama
    if (!nameField.value.trim()) {
      showError('Hata', 'Lütfen adınızı giriniz');
      nameField.classList.add('is-invalid');
      
      // Hata mesajı DOM'a ekleme
      var nameError = document.getElementById('musteriAd-error');
      if (!nameError) {
        nameError = document.createElement('div');
        nameError.id = 'musteriAd-error';
        nameError.className = 'invalid-feedback';
        nameField.parentNode.appendChild(nameError);
      }
      nameError.textContent = 'Lütfen adınızı giriniz';
      
      isValid = false;
    } else {
      nameField.classList.remove('is-invalid');
      var nameError = document.getElementById('musteriAd-error');
      if (nameError) nameError.remove();
      
      // Ek isim format doğrulama (ad ve soyad)
      if (nameField.value.trim().split(' ').length < 2) {
        showError('Hata', 'Lütfen ad ve soyadınızı giriniz');
        nameField.classList.add('is-invalid');
        
        // Hata mesajı DOM'a ekleme
        var nameError = document.getElementById('musteriAd-error');
        if (!nameError) {
          nameError = document.createElement('div');
          nameError.id = 'musteriAd-error';
          nameError.className = 'invalid-feedback';
          nameField.parentNode.appendChild(nameError);
        }
        nameError.textContent = 'Lütfen ad ve soyadınızı giriniz';
        
        isValid = false;
      }
    }
    
    // Telefon doğrulama - geliştirildi
    if (!phoneField.value.trim()) {
      showError('Hata', 'Lütfen telefon numaranızı giriniz');
      phoneField.classList.add('is-invalid');
      
      var phoneError = document.getElementById('musteriTelefon-error');
      if (!phoneError) {
        phoneError = document.createElement('div');
        phoneError.id = 'musteriTelefon-error';
        phoneError.className = 'invalid-feedback';
        phoneField.parentNode.appendChild(phoneError);
      }
      phoneError.textContent = 'Lütfen telefon numaranızı giriniz';
      
      isValid = false;
    } else {
      // Telefon formatını kontrol et (en az 10 rakam)
      var phoneDigits = phoneField.value.replace(/\D/g, '');
      if (phoneDigits.length < 10) {
        showError('Hata', 'Lütfen geçerli bir telefon numarası giriniz');
        phoneField.classList.add('is-invalid');
        
        var phoneError = document.getElementById('musteriTelefon-error');
        if (!phoneError) {
          phoneError = document.createElement('div');
          phoneError.id = 'musteriTelefon-error';
          phoneError.className = 'invalid-feedback';
          phoneField.parentNode.appendChild(phoneError);
        }
        phoneError.textContent = 'Lütfen geçerli bir telefon numarası giriniz';
        
        isValid = false;
      } else {
        phoneField.classList.remove('is-invalid');
        var phoneError = document.getElementById('musteriTelefon-error');
        if (phoneError) phoneError.remove();
      }
    }
  }
  
  // Step 2: Hizmet ve personel seçimi
  else if (currentTab === 1) {
    var serviceField = document.getElementById('service');
    var providerList = document.getElementById('providerList');
    
    // Hizmet seçimi doğrulama
    if (!serviceField || serviceField.value === '') {
      showError('Hata', 'Lütfen hizmet seçiniz');
      serviceField.classList.add('is-invalid');
      isValid = false;
    } else {
      serviceField.classList.remove('is-invalid');
    }
    
    // Personel seçimi doğrulama
    var selectedProvider = providerList.querySelector('input[type="radio"]:checked');
    if (!selectedProvider) {
      showError('Hata', 'Lütfen personel seçiniz');
      isValid = false;
    }
  }
  
  // Step 3: Tarih ve saat seçimi
  else if (currentTab === 2) {
    var dateField = document.getElementById('start-date');
    var timeSelect = document.getElementById('timeSelect');
    
    // Tarih seçimi doğrulama
    if (!dateField || !dateField.value) {
      showError('Hata', 'Lütfen tarih seçiniz');
      dateField.classList.add('is-invalid');
      isValid = false;
    } else {
      dateField.classList.remove('is-invalid');
    }
    
    // Saat seçimi doğrulama
    var selectedTime = timeSelect.querySelector('input[type="radio"]:checked');
    if (!selectedTime) {
      showError('Hata', 'Lütfen saat seçiniz');
      isValid = false;
    }
  }
  
  return isValid;
}

// Form gönderim işlemi geliştirildi - Added in v3.1.9
function submitForm() {
  console.log('Starting form submission process');
  
  // Son doğrulama
  if (!validateCurrentStep()) {
    console.warn('Son doğrulama başarısız, form gönderilmedi');
    return;
  }
  
  // Gönder butonunu al
  var btn = document.getElementById('randevuEkleBtn');
  if (!btn) {
    console.error('Gönder butonu bulunamadı!');
    return;
  }
  
  // Butonda yükleniyor durumunu göster
  btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> İşleniyor';
  btn.disabled = true;
  btn.classList.add('processing');
  
  // Yükleniyor göstergesi
  showLoading('Randevunuz oluşturuluyor, lütfen bekleyiniz...');
  
  // Form verilerini al
  var form = document.getElementById('wizard');
  var formData = new FormData(form);
  
  // İstek zaman aşımı kontrolü ekle
  window.requestTimeout = setTimeout(function() {
    btn.disabled = false;
    btn.innerHTML = 'Oluştur <i class="fas fa-check ms-2"></i>';
    btn.classList.remove('processing');
    
    Swal.fire({
      title: 'Hata',
      text: 'İstek zaman aşımına uğradı. Lütfen internet bağlantınızı kontrol edip tekrar deneyiniz.',
      icon: 'error',
      confirmButtonColor: '#ff950e',
      confirmButtonText: 'Tamam'
    });
  }, 30000); // 30 saniye sonra timeout
  
  // Ödeme yöntemi kontrolü
  var paymentMethodElement = document.getElementById('odemeYontemi');
  var paymentMethod = paymentMethodElement ? paymentMethodElement.value : 'Nakit';
  
  // Ödeme türüne göre farklı endpoint kullan
  if (paymentMethod === 'KrediKarti') {
    submitCreditCardPayment(formData, btn);
  } else {
    submitRegularForm(formData, btn);
  }
}

// Standart form gönderimi - Added in v3.1.9
function submitRegularForm(formData, btn) {
  console.log('Standart form gönderiliyor');
  
  // Timeout temizle (başarılı gönderimde)
  if (window.requestTimeout) {
    clearTimeout(window.requestTimeout);
  }
  
  // Fetch API ile form gönderimi
  fetch('/admin/system/settings/eral.php', {
    method: 'POST',
    body: formData,
    cache: 'no-cache', // Önbelleğe alma
    redirect: 'follow' // Yönlendirmeleri takip et
  })
  .then(function(response) {
    if (!response.ok) {
      throw new Error('Network response was not ok: ' + response.status);
    }
    return response.text();
  })
  .then(function(data) {
    console.log('Form yanıtı:', data);
    
    if (data === 'varvar') {
      Swal.fire({
        title: translations.sorry || "Üzgünüz",
        text: translations.otherselect || "Lütfen başka bir seçim yapınız",
        icon: 'error',
        confirmButtonColor: '#ff950e',
        confirmButtonText: translations.okey || "Tamam"
      }).then(function() {
        location.reload();
      });
    } else {
      // Başarılı yanıt için ek güvenlik kontrolü
      if (data.indexOf('<script>') !== -1 || data.indexOf('<iframe') !== -1) {
        throw new Error('Güvenlik uyarısı: Beklenmeyen içerik');
      }
      
      var paymentMethodElement = document.getElementById('odemeYontemi');
      var paymentMethod = paymentMethodElement ? paymentMethodElement.value : 'Nakit';
      
      if (paymentMethod === 'Havale') {
        location.href = "hsuccess";
      } else {
        location.href = "success";
      }
    }
  })
  .catch(function(error) {
    console.error('Form submission error:', error);
    
    // Butonu normal haline getir
    btn.disabled = false;
    btn.innerHTML = 'Oluştur <i class="fas fa-check ms-2"></i>';
    btn.classList.remove('processing');
    
    // Hata mesajı göster
    Swal.fire({
      title: translations.error || "Hata",
      text: translations.tryagain || "Bir hata oluştu. Lütfen tekrar deneyiniz.",
      icon: 'error',
      confirmButtonColor: '#ff950e',
      confirmButtonText: translations.okey || "Tamam"
    });
  });
}

// Geliştirilmiş hata mesajı - Updated in v3.1.9
function showError(title, message) {
  console.error(title + ': ' + message);
  
  Swal.fire({
    title: title,
    text: message,
    icon: 'error',
    confirmButtonColor: '#ff950e',
    confirmButtonText: 'Tamam',
    customClass: {
      popup: 'animated fadeInDown faster'
    }
  });
}

// Yükleniyor mesajı - Added in v3.1.9
function showLoading(message) {
  Swal.fire({
    title: 'İşleniyor',
    text: message,
    allowOutsideClick: false,
    allowEscapeKey: false,
    allowEnterKey: false,
    didOpen: () => {
      Swal.showLoading();
    }
  });
}
  console.log('Translations initialized');